#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/ads/adsapi.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::ads;

namespace ttv {
void Print(const Ad& ad, std::string indent) {
  std::cout << indent << "Ad:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(ad, identifier) << std::endl;
  std::cout << indent << PRINT_FIELD(ad, sequenceNumber) << std::endl;
  std::cout << indent << PRINT_FIELD(ad, adTitle) << std::endl;

  // TODO: We could print more
}

void Print(const std::vector<Ad>& ads, std::string indent) {
  std::cout << indent << "Ads:" << std::endl;
  indent += "  ";

  for (const auto& ad : ads) {
    Print(ad, indent);
  }
}
}  // namespace ttv

namespace {
const char* kTraceComponents[] = {"AdsAPI"};

std::shared_ptr<AdsAPI> gAdsAPI;
std::shared_ptr<Context> gContext;  // Cached by the static initializer

std::shared_ptr<CommandCategory> RegisterAdsCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Ads", "The ttv::ads::AdsAPI command interface");

  category->AddCommand("AdsInit")
    .AddFunction()
    .Description("Calls AdsAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gAdsAPI->Initialize([](TTV_ErrorCode ec) {
        std::cout << "AdsInit completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("AdsShutdown")
    .AddFunction()
    .Description("Calls AdsAPI::Shutdown")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gAdsAPI->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "AdsShutdown completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("AdsFetch")
    .AddFunction()
    .Description("Calls AdsAPI::FetchAds")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      // TODO: Populate these structs from the command line
      AdConfiguration config;
      config.languageCode = "en";
      config.adTrackingIdentifer = "";
      config.userAgentString = "TestUserAgent";
      config.adUnit = "/3576121/twitch.m/ios";
      config.mediaTypeFilters = {"video/mp4"};

      AdFetchRequestInfo info;
      info.channelName = "twitch";
      info.game = "Minecraft";
      info.lengthInSeconds = 60;
      info.position = AdPosition::Preroll;
      info.ppid = "optout";
      info.source = "DFP";
      info.vodID = "";
      info.vodName = "";
      info.vodType = "";

      TTV_ErrorCode ec = gAdsAPI->FetchAds(config, info, [](TTV_ErrorCode ec, std::vector<Ad>&& result) {
        if (TTV_SUCCEEDED(ec)) {
          Print(result, "  ");
        } else {
          std::cout << "AdsAPI::FetchAds async failure: " << ErrorToString(ec) << std::endl;
        }
      });

      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("AdsReportAdEvent")
    .AddFunction()
    .Description("Calls AdsAPI::ReportAdEvent")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::string url =
        "https://video-ad-stats.googlesyndication.com/video/client_events?event=2&web_property=ca-pub-8677887578896929&cpn=[CPN]&break_type=[BREAK_TYPE]&slot_pos=[SLOT_POS]&ad_id=[AD_ID]&ad_sys=[AD_SYS]&ad_len=[AD_LEN]&p_w=[P_W]&p_h=[P_H]&mt=[MT]&rwt=[RWT]&wt=[WT]&sdkv=[SDKV]&vol=[VOL]&content_v=[CONTENT_V]&conn=[CONN]&format=[FORMAT_NAMESPACE]_[FORMAT_TYPE]_[FORMAT_SUBTYPE]";
      std::map<std::string, std::string> macroSubstitutions = {
        {"cpn", "Hello"}, {"break_type", "Nap Time"}, {"slot_pos", "Vegas!"}, {"ad_id", "123456"},
        {"ad_sys", "Who knows!!!"}, {"ad_len", "60"}, {"p_w", "1024"}, {"p_h", "768"}
        // Let the rest of them go to default values
      };

      // use default user agent
      TTV_ErrorCode ec = gAdsAPI->ReportAdEvent(url, macroSubstitutions, "", [](TTV_ErrorCode ec) {
        if (TTV_SUCCEEDED(ec)) {
          // Request succeeded
        } else {
          std::cout << "AdsAPI::ReportAdEvent async failure: " << ErrorToString(ec) << std::endl;
        }
      });

      ReportCommandResult(ec);

      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      gContext = context;

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));
      gAdsAPI = std::make_shared<AdsAPI>();
      gAdsAPI->SetCoreApi(coreApi);
      context->RegisterModule(gAdsAPI);

      auto commands = RegisterAdsCommands(context);
      context->RegisterCommandCategory(commands);
    },
    1);
});
}  // namespace
