/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/ads/internal/pch.h"

#include "twitchsdk/ads/adsapi.h"

#include "twitchsdk/ads/internal/task/fetchadstask.h"
#include "twitchsdk/ads/internal/task/reportadeventtask.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/sha1.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

namespace {
using namespace ttv;
using namespace ttv::ads;

const char* kModuleName = "ttv::ads::AdsAPI";

TTV_ErrorCode InitiateAdFetch(const std::shared_ptr<TaskRunner>& taskRunner,
  const std::shared_ptr<FetchAdsTask::Context>& context,
  const std::function<void(TTV_ErrorCode ec)>& completeCallback) {
  auto task = std::make_shared<FetchAdsTask>(
    context, [taskRunner, context, completeCallback](FetchAdsTask* /*source*/, TTV_ErrorCode ec) {
      if (TTV_SUCCEEDED(ec)) {
        if (context->RequiresIteration()) {
          ec = InitiateAdFetch(taskRunner, context, completeCallback);
        } else {
          completeCallback(ec);
        }
      }

      if (TTV_FAILED(ec)) {
        completeCallback(ec);
      }
    });

  bool submitted = taskRunner->AddTask(task);
  if (submitted) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUTTING_DOWN;
  }
}
}  // namespace

ttv::ads::AdsAPI::AdsAPI() {}

std::string ttv::ads::AdsAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::ads::AdsAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  TTV_ASSERT(coreApi != nullptr);

  mCoreApi = coreApi;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::ads::AdsAPI::Initialize(const InitializeCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);
  TTV_RETURN_ON_NULL(mCoreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;

    mCoreApiClient = std::make_shared<CoreApiClient>();
    mMainEventScheduler = mCoreApi->GetMainEventScheduler();
    mTaskRunner = std::make_shared<TaskRunner>("AdsAPI");
  }

  ec = mCoreApi->RegisterClient(mCoreApiClient);
  if (TTV_SUCCEEDED(ec)) {
    NotifyStateChange();
    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::ads::AdsAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);

  if (TTV_SUCCEEDED(ec)) {
    RegisterShutdownCallback(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::ads::AdsAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    case State::Initialized: {
      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::ads::AdsAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::ads::AdsAPI::CompleteShutdown() {
  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  if (mCoreApiClient != nullptr) {
    if (mCoreApi != nullptr) {
      mCoreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  mCoreApi.reset();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::ads::AdsAPI::FetchAds(
  const AdConfiguration& config, const AdFetchRequestInfo& info, const FetchAdsCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(callback, TTV_EC_INVALID_ARG);

  auto context = std::make_shared<FetchAdsTask::Context>();
  context->config = config;
  context->info = info;

  TTV_ErrorCode ec = InitiateAdFetch(mTaskRunner, context, [callback, context](TTV_ErrorCode callbackEc) {
    std::vector<Ad> result;

    if (TTV_SUCCEEDED(callbackEc)) {
      result = std::move(context->ads);

      // Apply filters if necessary
      if (!context->config.mediaTypeFilters.empty()) {
        for (Ad& ad : result) {
          for (LinearCreative& creative : ad.linearCreatives) {
            creative.mediaFiles.erase(std::remove_if(creative.mediaFiles.begin(), creative.mediaFiles.end(),
                                        [context](const MediaFile& mediaFile) {
                                          return std::find(context->config.mediaTypeFilters.begin(),
                                                   context->config.mediaTypeFilters.end(),
                                                   mediaFile.type) == context->config.mediaTypeFilters.end();
                                        }),
              creative.mediaFiles.end());
          }

          // Remove all empty linear creatives
          ad.linearCreatives.erase(std::remove_if(ad.linearCreatives.begin(), ad.linearCreatives.end(),
                                     [context](const LinearCreative& creative) { return creative.mediaFiles.empty(); }),
            ad.linearCreatives.end());
        }

        // Remove all empty ads
        result.erase(
          std::remove_if(result.begin(), result.end(), [context](const Ad& ad) { return ad.linearCreatives.empty(); }),
          result.end());
      }

      // Sort media files by decreasing width
      for (Ad& ad : result) {
        for (LinearCreative& creative : ad.linearCreatives) {
          std::sort(creative.mediaFiles.begin(), creative.mediaFiles.end(),
            [](const MediaFile& a, const MediaFile& b) { return a.width > b.width; });
        }
      }
    }

    callback(callbackEc, std::move(result));
  });

  return ec;
}

TTV_ErrorCode ttv::ads::AdsAPI::ReportAdEvent(const std::string& url,
  const std::map<std::string, std::string>& macroSubstitutions, const std::string& adUserAgent,
  const ReportAdEventsCallback& callback) {
  auto task = std::make_shared<ReportAdEventTask>(
    url, macroSubstitutions, adUserAgent, [callback](ReportAdEventTask* /*source*/, TTV_ErrorCode ec) {
      if (callback != nullptr) {
        callback(ec);
      }
    });

  bool submitted = mTaskRunner->AddTask(task);
  if (submitted) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUTTING_DOWN;
  }
}

TTV_ErrorCode ttv::ads::AdsAPI::ReportAdEvents(const std::vector<std::string>& urls,
  const std::map<std::string, std::string>& macroSubstitutions, const std::string& adUserAgent) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  for (const auto& url : urls) {
    ReportAdEvent(url, macroSubstitutions, adUserAgent, nullptr);
  }

  return ec;
}

std::string ttv::ads::AdsAPI::CoreApiClient::GetClientName() {
  return kModuleName;
}
