/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/ads/internal/pch.h"

#include "twitchsdk/ads/internal/stringutil.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/random.h"
#include "twitchsdk/core/utf8.h"

#include <time.h>

#include <map>
#include <sstream>

std::string ttv::ads::SanitizeString(const std::string& string) {
  std::ostringstream stringStream;
  bool isInNonAlphanumericSequence = false;

  auto currentCharacter = string.c_str();
  while (*currentCharacter != '\0') {
    int sequenceLength;
    auto nextCharacter = AdvanceToNextUtf8Character(currentCharacter, sequenceLength);

    if (sequenceLength <= 1 && isalnum(static_cast<unsigned char>(*currentCharacter))) {
      isInNonAlphanumericSequence = false;
      stringStream << static_cast<char>(tolower(*currentCharacter));
    } else if (!isInNonAlphanumericSequence) {
      isInNonAlphanumericSequence = true;
      stringStream << "_";
    }

    currentCharacter = nextCharacter;
  }

  return stringStream.str();
}

std::string ttv::ads::SubstituteMacrosInUrl(
  const std::string& url, const std::map<std::string, std::string>& substitutions) {
  size_t currentIndex = 0;
  std::ostringstream stringStream;

  while (currentIndex < url.size()) {
    auto startPosition = url.find('[', currentIndex);

    if (startPosition == std::string::npos) {
      // Write the remainder of the URL
      stringStream.write(url.data() + currentIndex, static_cast<std::streamsize>(url.size() - currentIndex));
      break;
    } else {
      auto dataPointer = url.data();
      stringStream.write(dataPointer + currentIndex, static_cast<std::streamsize>(startPosition - currentIndex));

      auto endPosition = url.find(']', startPosition + 1);

      if (endPosition == std::string::npos) {
        // Malformed. Bail and return an empty string.
        return "";
      } else {
        std::ostringstream macroStream;

        for (size_t macroIndex = startPosition + 1; macroIndex < endPosition; macroIndex++) {
          macroStream << static_cast<char>(tolower(dataPointer[macroIndex]));
        }

        auto macroKey = macroStream.str();
        auto iter = substitutions.find(macroKey);

        if (iter != substitutions.end()) {
          stringStream << UrlEncode(iter->second);
        } else if (macroKey == "cachebusting") {
          // The VAST standard calls for a random 8 digit number here.
          uint32_t min = 10000000;
          uint32_t max = 99999999;
          std::uniform_int_distribution<uint32_t> distribution(min, max);
          uint32_t vastRandomNumericSequence = distribution(ttv::random::GetGenerator());
          stringStream << vastRandomNumericSequence;
        }

        currentIndex = endPosition + 1;
      }
    }
  }

  return stringStream.str();
}
