/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/ads/internal/pch.h"

#include "twitchsdk/ads/internal/task/fetchadstask.h"

#include "twitchsdk/ads/internal/stringutil.h"
#include "twitchsdk/ads/internal/vastparser.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/random.h"

#include <ctime>

namespace {
using namespace ttv;
using namespace ttv::ads;

const char* kDFPBaseURL = "https://pubads.g.doubleclick.net/gampad/ads";

std::string ContentModeToString(ContentMode contentMode) {
  switch (contentMode) {
    case ContentMode::Live:
    default:
      return "live";
    case ContentMode::Playlist:
      return "playlist";
    case ContentMode::Vod:
      return "vod";
    case ContentMode::VodCast:
      return "vodcast";
  }
}
}  // namespace

ttv::ads::FetchAdsTask::FetchAdsTask(const std::shared_ptr<Context>& context, const Callback& callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(callback), mContext(context), mRequestType(RequestType::Initial) {
  ttv::trace::Message("FetchAdsTask", MessageLevel::Info, "FetchAdsTask created");
}

std::string ads::FetchAdsTask::GenerateUrl() {
  std::vector<std::pair<std::string, std::string>> requestParameters;
  requestParameters.emplace_back("sz", "640x480");
  requestParameters.emplace_back("gdfp_req", "1");
  requestParameters.emplace_back("env", "vp");
  requestParameters.emplace_back("output", "xml_vast2");
  requestParameters.emplace_back("unviewed_position_start", "1");
  requestParameters.emplace_back("impl", "s");
  requestParameters.emplace_back("url", "http://www.twitch.tv");
  requestParameters.emplace_back("iu", mContext->config.adUnit);
  requestParameters.emplace_back("ad_rule", "1");

  uint32_t min = 0;
  uint32_t max = 99999999;
  std::uniform_int_distribution<uint32_t> distribution(min, max);
  uint32_t vastRandomNumericSequence = distribution(ttv::random::GetGenerator());
  requestParameters.emplace_back("correlator", std::to_string(vastRandomNumericSequence));

  std::vector<std::pair<std::string, std::string>> customParameters;

  customParameters.emplace_back("game", SanitizeString(mContext->info.game));
  customParameters.emplace_back("chan", mContext->info.channelName);
  customParameters.emplace_back("chanid", std::to_string(static_cast<int>(mContext->info.channelId)));
  customParameters.emplace_back("mature", mContext->info.mature ? "true" : "false");
  customParameters.emplace_back("live", mContext->info.live ? "true" : "false");
  customParameters.emplace_back("content_mode", ContentModeToString(mContext->info.contentMode));
  customParameters.emplace_back("language", mContext->config.languageCode);
  customParameters.emplace_back("pos", std::to_string(static_cast<int>(mContext->info.position)));
  customParameters.emplace_back("timebreak", std::to_string(mContext->info.lengthInSeconds));

  if (!mContext->info.vodType.empty()) {
    customParameters.emplace_back("vod_type", mContext->info.vodType);
  }
  if (!mContext->info.vodID.empty()) {
    customParameters.emplace_back("vod_id", mContext->info.vodID);
  }
  if (!mContext->info.vodName.empty()) {
    customParameters.emplace_back("vod_name", SanitizeString(mContext->info.vodName));
  }

  if (mContext->config.userAgentString.length() > 0) {
    customParameters.emplace_back("user_agent", mContext->config.userAgentString);
  }

  TTV_ASSERT(!mContext->config.platform.empty());
  if (mContext->config.platform.length() > 0) {
    customParameters.emplace_back("platform", mContext->config.platform);
  } else {
    ttv::trace::Message("FetchAdsTask", MessageLevel::Warning, "No platform specified for ad fetch.");
  }

  if (mContext->config.adTrackingIdentifer.length() > 0) {
    customParameters.emplace_back("adid", mContext->config.adTrackingIdentifer);
  } else {
    customParameters.emplace_back("adid", "optout");
  }

  if (mContext->info.ppid.length() > 0) {
    requestParameters.emplace_back("ppid", mContext->info.ppid);
    customParameters.emplace_back("ppid", mContext->info.ppid);
  } else {
    customParameters.emplace_back("ppid", "optout");
  }

  requestParameters.emplace_back("cust_params", BuildUrlEncodedRequestParams(customParameters));

  std::string url = kDFPBaseURL;
  url.append("?");
  url.append(BuildUrlEncodedRequestParams(requestParameters));

  return url;
}

void ttv::ads::FetchAdsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  // The initial request it created from the input parameters
  if (mContext->initialRequest) {
    mContext->initialRequest = false;
    requestInfo.url = GenerateUrl();
    mRequestType = RequestType::Initial;
  }
  // Subsequent requests are created from urls in previous responses
  else if (!mContext->playlistUrls.empty()) {
    requestInfo.url = mContext->playlistUrls.back();
    mContext->playlistUrls.pop_back();
    mRequestType = RequestType::Playlist;
  } else if (!mContext->errorUrls.empty()) {
    requestInfo.url = mContext->errorUrls.back();
    mContext->errorUrls.pop_back();
    mRequestType = RequestType::Error;
  } else if (!mContext->adWrappers.empty()) {
    mActiveWrapper = std::move(mContext->adWrappers.back());
    requestInfo.url = mActiveWrapper.nextTagUrl;
    mContext->adWrappers.pop_back();
    mRequestType = RequestType::Wrapper;
  } else {
    TTV_ASSERT(false);
  }

  if (!mContext->config.userAgentString.empty()) {
    requestInfo.requestHeaders.emplace_back("User-Agent", mContext->config.userAgentString);
  }

  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::ads::FetchAdsTask::ProcessResponse(uint statusCode, const std::vector<char>& response) {
  switch (mRequestType) {
    case RequestType::Initial:
    case RequestType::Playlist: {
      if (Is2XX(statusCode)) {
        std::string body(response.begin(), response.end());
        auto parseError = ParseVASTResponse(body, mContext->ads, mContext->adWrappers, mContext->playlistUrls);
        if (parseError != AdError::None) {
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        }
      } else {
        mTaskStatus = TTV_EC_API_REQUEST_FAILED;
      }

      break;
    }
    case RequestType::Wrapper: {
      if (Is2XX(statusCode)) {
        std::string body(response.begin(), response.end());
        auto error = ParseVASTResponseFromWrapper(body, mContext->ads, mContext->adWrappers, mActiveWrapper);
        if (error != AdError::None) {
          for (const auto& errorUrl : mActiveWrapper.errorUrls) {
            std::map<std::string, std::string> macroSubstitutions;
            macroSubstitutions["errorcode"] = std::to_string(static_cast<std::underlying_type<AdError>::type>(error));

            mContext->errorUrls.push_back(SubstituteMacrosInUrl(errorUrl, macroSubstitutions));
          }
        }
      } else {
        for (const auto& errorUrl : mActiveWrapper.errorUrls) {
          std::map<std::string, std::string> macroSubstitutions;
          macroSubstitutions["errorcode"] =
            std::to_string(static_cast<std::underlying_type<AdError>::type>(AdError::WrapperFailed));

          mContext->errorUrls.push_back(SubstituteMacrosInUrl(errorUrl, macroSubstitutions));
        }
      }

      break;
    }
    case RequestType::Error: {
      // We don't care about the response
      break;
    }
  }
}

void ttv::ads::FetchAdsTask::ResponseCallback(uint statusCode, const std::vector<char>& response) {
  if (!mAborted) {
    mTaskStatus = TTV_EC_SUCCESS;

    // Process the response regardless of the statusCode
    ProcessResponse(statusCode, response);
  }
}

void ttv::ads::FetchAdsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec);
  }
}
