/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/ads/internal/pch.h"

#include "twitchsdk/ads/internal/vastparser.h"

#include "twitchsdk/ads/internal/stringutil.h"
#include "twitchsdk/core/xml/tinyxml2.h"

#include <inttypes.h>

#include <algorithm>
#include <iterator>

namespace {
using namespace ttv::ads;

const uint8_t kMaxWrapperDepth = 4;

bool SupportsVASTVersion(const std::string& version) {
  return ((version == "2.0") || (version == "2.0.1") || (version == "3.0"));
}

bool ParseTextElement(const ttv::xml::XMLElement* element, std::string& result) {
  if (element != nullptr) {
    auto text = element->GetText();
    if (text != nullptr) {
      result = text;
      return true;
    }
  }

  return false;
}

bool SetFromAttribute(const ttv::xml::XMLElement* element, const std::string& attribute, std::string& result) {
  if (element != nullptr) {
    auto att = element->Attribute(attribute.c_str());
    if (att != nullptr) {
      result = att;
      return true;
    }
  }

  return false;
}

AdError ParseDuration(const char* timeString, uint32_t& durationSeconds) {
  int hours;
  int minutes;
  int seconds;

#ifdef _MSC_VER

  if (sscanf_s(timeString, "%2d:%2d:%2d", &hours, &minutes, &seconds) == 3) {
#else

  if (sscanf(timeString, "%2d:%2d:%2d", &hours, &minutes, &seconds) == 3) {
#endif
    durationSeconds = static_cast<uint32_t>((((hours * 60) + minutes) * 60) + seconds);
    return AdError::None;
  } else {
    return AdError::VASTSchemaValidationError;
  }
}

AdError ParseMediaFile(const ttv::xml::XMLElement* mediaFileElement, MediaFile& mediaFile) {
  mediaFile.maintainAspectRatio = mediaFileElement->BoolAttribute("maintainAspectRatio");
  mediaFile.scalable = mediaFileElement->BoolAttribute("scalable");
  mediaFile.height = mediaFileElement->UnsignedAttribute("height");
  mediaFile.width = mediaFileElement->UnsignedAttribute("width");
  mediaFile.bitrate = mediaFileElement->UnsignedAttribute("bitrate");

  SetFromAttribute(mediaFileElement, "type", mediaFile.type);
  SetFromAttribute(mediaFileElement, "delivery", mediaFile.delivery);
  SetFromAttribute(mediaFileElement, "id", mediaFile.identifier);
  ParseTextElement(mediaFileElement, mediaFile.resourceUrl);

  return AdError::None;
}

TrackingEventType EventTypeForString(const std::string& eventString) {
  TrackingEventType eventType = TrackingEventType::Unknown;

  if (eventString == "creativeView") {
    eventType = TrackingEventType::CreativeView;
  } else if (eventString == "start") {
    eventType = TrackingEventType::Start;
  } else if (eventString == "firstQuartile") {
    eventType = TrackingEventType::FirstQuartile;
  } else if (eventString == "midpoint") {
    eventType = TrackingEventType::Midpoint;
  } else if (eventString == "thirdQuartile") {
    eventType = TrackingEventType::ThirdQuartile;
  } else if (eventString == "complete") {
    eventType = TrackingEventType::Complete;
  } else if (eventString == "mute") {
    eventType = TrackingEventType::Mute;
  } else if (eventString == "unmute") {
    eventType = TrackingEventType::Unmute;
  } else if (eventString == "pause") {
    eventType = TrackingEventType::Pause;
  } else if (eventString == "rewind") {
    eventType = TrackingEventType::Rewind;
  } else if (eventString == "resume") {
    eventType = TrackingEventType::Resume;
  } else if (eventString == "fullscreen") {
    eventType = TrackingEventType::Fullscreen;
  } else if (eventString == "exitFullscreen") {
    eventType = TrackingEventType::ExitFullscreen;
  } else if (eventString == "expand") {
    eventType = TrackingEventType::Expand;
  } else if (eventString == "collapse") {
    eventType = TrackingEventType::Collapse;
  } else if (eventString == "acceptInvitation") {
    eventType = TrackingEventType::AcceptInvitation;
  } else if (eventString == "close") {
    eventType = TrackingEventType::Close;
  } else if (eventString == "skip") {
    eventType = TrackingEventType::Skip;
  } else if (eventString == "progress") {
    eventType = TrackingEventType::Progress;
  }

  return eventType;
}

AdError ParseTrackingEvent(const ttv::xml::XMLElement* trackingEventElement,
  std::map<TrackingEventType, std::vector<std::string>>& trackingEventUrls) {
  auto eventString = trackingEventElement->Attribute("event");
  auto trackingUrl = trackingEventElement->GetText();

  if ((eventString == nullptr) || (trackingUrl == nullptr)) {
    return AdError::VASTSchemaValidationError;
  }

  auto eventType = EventTypeForString(eventString);
  trackingEventUrls[eventType].push_back(trackingUrl);

  return AdError::None;
}

AdError ParseLinearCreative(const ttv::xml::XMLElement* linearElement, LinearCreative& linearCreative) {
  auto durationElement = linearElement->FirstChildElement("Duration");

  if (durationElement == nullptr) {
    return AdError::VASTSchemaValidationError;
  }

  auto durationString = durationElement->GetText();

  if (durationString == nullptr) {
    return AdError::VASTSchemaValidationError;
  }

  auto error = ParseDuration(durationString, linearCreative.durationSeconds);

  if (error != AdError::None) {
    return error;
  }

  auto trackingEventsElement = linearElement->FirstChildElement("TrackingEvents");

  if (trackingEventsElement != nullptr) {
    auto trackingElement = trackingEventsElement->FirstChildElement("Tracking");

    while (trackingElement != nullptr) {
      ParseTrackingEvent(trackingElement, linearCreative.trackingEventUrls);

      trackingElement = trackingElement->NextSiblingElement("Tracking");
    }
  }

  auto videoClicksElement = linearElement->FirstChildElement("VideoClicks");

  if (videoClicksElement != nullptr) {
    ParseTextElement(videoClicksElement->FirstChildElement("ClickThrough"), linearCreative.clickThroughUrl);

    auto clickTrackingElement = videoClicksElement->FirstChildElement("ClickTracking");

    while (clickTrackingElement != nullptr) {
      linearCreative.clickTrackingUrls.emplace_back();

      if (!ParseTextElement(clickTrackingElement, linearCreative.clickTrackingUrls.back())) {
        linearCreative.clickTrackingUrls.pop_back();
      }

      clickTrackingElement = clickTrackingElement->NextSiblingElement("ClickTracking");
    }
  }

  auto adParametersElement = linearElement->FirstChildElement("AdParameters");

  if (adParametersElement != nullptr) {
    // We don't currently support xml encoded ad parameters.
    bool isXmlEncoded = adParametersElement->BoolAttribute("xmlEncoded");

    if (!isXmlEncoded) {
      ParseTextElement(adParametersElement, linearCreative.adParametersString);
    }
  }

  auto mediaFilesElement = linearElement->FirstChildElement("MediaFiles");

  if (mediaFilesElement != nullptr) {
    auto mediaFileElement = mediaFilesElement->FirstChildElement("MediaFile");

    while (mediaFileElement != nullptr) {
      linearCreative.mediaFiles.emplace_back();

      if (ParseMediaFile(mediaFileElement, linearCreative.mediaFiles.back()) != AdError::None) {
        linearCreative.mediaFiles.pop_back();
      }

      mediaFileElement = mediaFileElement->NextSiblingElement("MediaFile");
    }
  }

  return AdError::None;
}

AdError ParseCompanionCreative(const ttv::xml::XMLElement* companionElement, CompanionCreative& companionCreative) {
  companionCreative.height = companionElement->UnsignedAttribute("height");
  companionCreative.width = companionElement->UnsignedAttribute("width");

  SetFromAttribute(companionElement, "id", companionCreative.identifier);

  auto staticResourceElement = companionElement->FirstChildElement("StaticResource");

  if (staticResourceElement != nullptr) {
    SetFromAttribute(staticResourceElement, "creativeType", companionCreative.creativeType);
    ParseTextElement(staticResourceElement, companionCreative.resourceUrl);
  }

  auto trackingEventsElement = companionElement->FirstChildElement("TrackingEvents");

  if (trackingEventsElement != nullptr) {
    auto trackingEventElement = trackingEventsElement->FirstChildElement("Tracking");

    while (trackingEventElement != nullptr) {
      ParseTrackingEvent(trackingEventElement, companionCreative.trackingEventUrls);

      trackingEventElement = trackingEventElement->NextSiblingElement("Tracking");
    }
  }

  return AdError::None;
}

AdError ParseInlineAd(const ttv::xml::XMLElement* adElement, Ad& ad) {
  TTV_ASSERT(adElement != nullptr);

  ParseTextElement(adElement->FirstChildElement("AdSystem"), ad.adSystem);
  ParseTextElement(adElement->FirstChildElement("AdTitle"), ad.adTitle);
  ParseTextElement(adElement->FirstChildElement("Description"), ad.adDescription);

  ad.errorUrls.emplace_back();

  if (!ParseTextElement(adElement->FirstChildElement("Error"), ad.errorUrls.back())) {
    ad.errorUrls.pop_back();
  }

  auto impressionElement = adElement->FirstChildElement("Impression");

  while (impressionElement != nullptr) {
    Impression impression;

    SetFromAttribute(impressionElement, "id", impression.identifier);

    if (ParseTextElement(impressionElement, impression.resourceUrl)) {
      ad.impressions.emplace_back(impression);
    }

    impressionElement = impressionElement->NextSiblingElement("Impression");
  }

  auto creativesElement = adElement->FirstChildElement("Creatives");

  if (creativesElement != nullptr) {
    auto creativeElement = creativesElement->FirstChildElement("Creative");

    while (creativeElement != nullptr) {
      auto linearElement = creativeElement->FirstChildElement("Linear");

      if (linearElement != nullptr) {
        ad.linearCreatives.emplace_back();

        if (ParseLinearCreative(linearElement, ad.linearCreatives.back()) != AdError::None) {
          ad.linearCreatives.pop_back();
        }
      } else {
        auto companionAdsElement = creativeElement->FirstChildElement("CompanionAds");

        if (companionAdsElement != nullptr) {
          auto companionElement = companionAdsElement->FirstChildElement("Companion");

          while (companionElement != nullptr) {
            ad.companionCreatives.emplace_back();

            if (ParseCompanionCreative(companionElement, ad.companionCreatives.back()) != AdError::None) {
              ad.companionCreatives.pop_back();
            }

            companionElement = companionElement->NextSiblingElement("Companion");
          }
        }
      }

      creativeElement = creativeElement->NextSiblingElement("Creative");
    }
  }

  return AdError::None;
}

AdError ParseAdWrapper(const ttv::xml::XMLElement* adElement, AdWrapper& adWrapper) {
  auto tagURIElement = adElement->FirstChildElement("VASTAdTagURI");

  if (tagURIElement == nullptr) {
    tagURIElement = adElement->FirstChildElement("VASTAdTagURL");
  }

  ParseTextElement(tagURIElement, adWrapper.nextTagUrl);

  adWrapper.errorUrls.emplace_back();

  if (!ParseTextElement(adElement->FirstChildElement("Error"), adWrapper.errorUrls.back())) {
    adWrapper.errorUrls.pop_back();
  }

  auto impressionElement = adElement->FirstChildElement("Impression");

  while (impressionElement != nullptr) {
    Impression impression;

    SetFromAttribute(impressionElement, "id", impression.identifier);

    if (ParseTextElement(impressionElement, impression.resourceUrl)) {
      adWrapper.impressions.emplace_back(impression);
    }

    impressionElement = impressionElement->NextSiblingElement("Impression");
  }

  auto creativesElement = adElement->FirstChildElement("Creatives");

  if (creativesElement != nullptr) {
    auto creativeElement = creativesElement->FirstChildElement("Creative");

    while (creativeElement != nullptr) {
      auto linearElement = creativeElement->FirstChildElement("Linear");

      if (linearElement != nullptr) {
        auto trackingEventsElement = linearElement->FirstChildElement("TrackingEvents");

        if (trackingEventsElement != nullptr) {
          auto trackingElement = trackingEventsElement->FirstChildElement("Tracking");

          while (trackingElement != nullptr) {
            ParseTrackingEvent(trackingElement, adWrapper.linearTrackingEventUrls);

            trackingElement = trackingElement->NextSiblingElement("Tracking");
          }
        }

        auto videoClicksElement = linearElement->FirstChildElement("VideoClicks");

        if (videoClicksElement != nullptr) {
          auto clickTrackingElement = videoClicksElement->FirstChildElement("ClickTracking");

          while (clickTrackingElement != nullptr) {
            adWrapper.linearClickTrackingUrls.emplace_back();

            if (!ParseTextElement(clickTrackingElement, adWrapper.linearClickTrackingUrls.back())) {
              adWrapper.linearClickTrackingUrls.pop_back();
            }

            clickTrackingElement = clickTrackingElement->NextSiblingElement("ClickTracking");
          }
        }
      } else {
        auto companionAdsElement = creativeElement->FirstChildElement("CompanionAds");

        if (companionAdsElement != nullptr) {
          auto companionElement = creativesElement->FirstChildElement("Companion");

          while (companionElement != nullptr) {
            adWrapper.companionCreatives.emplace_back();

            if (ParseCompanionCreative(companionElement, adWrapper.companionCreatives.back()) != AdError::None) {
              adWrapper.companionCreatives.pop_back();
            }

            companionElement = companionElement->NextSiblingElement("Companion");
          }
        }
      }

      creativeElement = creativeElement->NextSiblingElement("Creative");
    }
  }

  return AdError::None;
}

AdError ParseDFPPlaylist(const ttv::xml::XMLElement* playlistElement, std::vector<std::string>& playlistUrls) {
  auto positionElement = playlistElement->FirstChildElement();

  while (positionElement != nullptr) {
    auto positionString = positionElement->Name();

    if ((strcmp(positionString, "Preroll") == 0) || (strcmp(positionString, "Midroll") == 0) ||
        (strcmp(positionString, "Postroll") == 0)) {
      auto adElement = positionElement->FirstChildElement();

      while (adElement != nullptr) {
        auto url = adElement->GetText();

        if (url != nullptr) {
          playlistUrls.push_back(url);
        }

        adElement = adElement->NextSiblingElement();
      }
    }

    positionElement = positionElement->NextSiblingElement();
  }

  return AdError::None;
}
}  // namespace

ttv::ads::AdError ttv::ads::ParseVASTResponse(const std::string& responseData, std::vector<Ad>& inlineAds,
  std::vector<AdWrapper>& adWrappers, std::vector<std::string>& playlistUrls) {
  using namespace ttv::xml;

  XMLDocument document;

  if (document.Parse(responseData.data(), responseData.length()) != XML_SUCCESS) {
    return AdError::XMLParseError;
  }

  auto rootElement = document.RootElement();

  if (rootElement == nullptr) {
    return AdError::VASTSchemaValidationError;
  }

  if (strcmp(rootElement->Name(), "Playlist") == 0) {
    return ParseDFPPlaylist(rootElement, playlistUrls);
  }

  if (strcmp(rootElement->Name(), "VAST") != 0) {
    return AdError::VASTSchemaValidationError;
  }

  auto versionString = rootElement->Attribute("version");

  if (versionString == nullptr || !SupportsVASTVersion(versionString)) {
    return AdError::VASTUnsupportedVersion;
  }

  auto adElement = rootElement->FirstChildElement("Ad");
  if (adElement != nullptr) {
    auto identifier = adElement->Attribute("id");
    auto sequence = adElement->Attribute("sequence");
    uint32_t sequenceNumber = 0;
    if (sequence != nullptr) {
#ifdef _MSC_VER

      sscanf_s(sequence, "%" PRIu32, &sequenceNumber);
#else
      sscanf(sequence, "%" PRIu32, &sequenceNumber);
#endif
    }

    while (adElement != nullptr) {
      auto inlineElement = adElement->FirstChildElement("InLine");

      if (inlineElement != nullptr) {
        inlineAds.emplace_back();

        auto& ad = inlineAds.back();
        if (ParseInlineAd(inlineElement, ad) != AdError::None) {
          inlineAds.pop_back();
        } else {
          if (identifier != nullptr) {
            ad.identifier = identifier;
          }
          ad.sequenceNumber = sequenceNumber;
        }
      } else {
        auto wrapperElement = adElement->FirstChildElement("Wrapper");

        if (wrapperElement != nullptr) {
          adWrappers.emplace_back();

          if (ParseAdWrapper(wrapperElement, adWrappers.back()) != AdError::None) {
            adWrappers.pop_back();
          }
        }
      }

      adElement = adElement->NextSiblingElement("Ad");
    }
  }

  return AdError::None;
}

ttv::ads::AdError ttv::ads::ParseVASTResponseFromWrapper(const std::string& responseData, std::vector<Ad>& inlineAds,
  std::vector<AdWrapper>& adWrappers, AdWrapper& parentWrapper) {
  using namespace ttv::xml;

  XMLDocument document;

  if (document.Parse(responseData.data(), responseData.length()) != XML_SUCCESS) {
    return AdError::XMLParseError;
  }

  auto rootElement = document.RootElement();

  if ((rootElement == nullptr) || (strcmp(rootElement->Name(), "VAST") != 0)) {
    return AdError::VASTSchemaValidationError;
  }

  auto versionString = rootElement->Attribute("version");

  if (versionString == nullptr || !SupportsVASTVersion(versionString)) {
    return AdError::VASTUnsupportedVersion;
  }

  auto adElement = rootElement->FirstChildElement("Ad");

  if (adElement == nullptr) {
    return AdError::WrapperNoAdFound;
  }

  auto inlineElement = adElement->FirstChildElement("InLine");

  if (inlineElement != nullptr) {
    inlineAds.emplace_back();
    auto& inlineAd = inlineAds.back();
    auto parseError = ParseInlineAd(inlineElement, inlineAd);

    if (parseError != AdError::None) {
      inlineAds.pop_back();
      return parseError;
    }

    if (inlineAd.linearCreatives.size() > 0) {
      auto& linearCreative = inlineAd.linearCreatives.front();

      for (auto& eventPair : parentWrapper.linearTrackingEventUrls) {
        auto& eventUrls = linearCreative.trackingEventUrls[eventPair.first];
        std::move(eventPair.second.begin(), eventPair.second.end(), std::back_inserter(eventUrls));
      }

      std::move(parentWrapper.linearClickTrackingUrls.begin(), parentWrapper.linearClickTrackingUrls.end(),
        std::back_inserter(linearCreative.clickTrackingUrls));
    }

    std::move(parentWrapper.companionCreatives.begin(), parentWrapper.companionCreatives.end(),
      std::back_inserter(inlineAd.companionCreatives));
    std::move(parentWrapper.errorUrls.begin(), parentWrapper.errorUrls.end(), std::back_inserter(inlineAd.errorUrls));
    std::move(
      parentWrapper.impressions.begin(), parentWrapper.impressions.end(), std::back_inserter(inlineAd.impressions));

    return AdError::None;
  }

  auto wrapperElement = adElement->FirstChildElement("Wrapper");

  if (wrapperElement != nullptr) {
    parentWrapper.depth++;

    if (parentWrapper.depth < kMaxWrapperDepth) {
      AdError adError = ParseAdWrapper(wrapperElement, parentWrapper);
      if (!parentWrapper.nextTagUrl.empty()) {
        adWrappers.emplace_back(parentWrapper);
      }

      return adError;
    } else {
      return AdError::WrapperDepth;
    }
  }

  return AdError::WrapperNoAdFound;
}
