/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/adsapitest.h"
#include "testutilities.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::ads;
using namespace ttv::ads::test;

const uint32_t kTestTimeoutMilliseconds = 2000;

void SetAdConfiguration(AdConfiguration& config) {
  config.languageCode = "en";
  config.userAgentString = "TestUserAgent";
  config.adUnit = "/3576121/twitch.m/ios";
  config.adTrackingIdentifer = "";
  config.platform = "test_platform";
}

void SetSampleAdFetchRequest(AdFetchRequestInfo& requestInfo) {
  requestInfo.source = "DFP";
  requestInfo.channelName = "TestChannel";
  requestInfo.game = "Test Game";
  requestInfo.vodType = "";
  requestInfo.vodName = "";
  requestInfo.vodID = "";
  requestInfo.ppid = "optout";
  requestInfo.lengthInSeconds = 30;
}

void SetSamplePrerollAdFetchRequest(AdFetchRequestInfo& requestInfo) {
  SetSampleAdFetchRequest(requestInfo);
  requestInfo.position = AdPosition::Preroll;
}

void SetSampleMidrollAdFetchRequest(AdFetchRequestInfo& requestInfo) {
  SetSampleAdFetchRequest(requestInfo);
  requestInfo.position = ads::AdPosition::Midroll;
}

void VerifyBasicInfoFromInlineLinearSample(const ads::Ad& ad) {
  ASSERT_EQ(ad.adSystem, "SampleAdSystem");
  ASSERT_EQ(ad.adTitle, "Sample Ad Title");
  ASSERT_EQ(ad.adDescription, "Sample Ad Description");

  ASSERT_EQ(ad.linearCreatives.size(), 1U);
  const auto& linear = ad.linearCreatives[0];

  ASSERT_EQ(linear.durationSeconds, 30U);
  ASSERT_EQ(linear.clickThroughUrl, "https://sampleclickthrough.com");

  const auto& mediaFiles = linear.mediaFiles;
  ASSERT_EQ(mediaFiles.size(), 2U);

  const auto& firstMediaFile = mediaFiles[0];
  ASSERT_EQ(firstMediaFile.identifier, "mediafile1");
  ASSERT_EQ(firstMediaFile.delivery, "progressive");
  ASSERT_EQ(firstMediaFile.width, 640U);
  ASSERT_EQ(firstMediaFile.height, 480U);
  ASSERT_EQ(firstMediaFile.type, "video/3gpp");
  ASSERT_EQ(firstMediaFile.bitrate, 400U);
  ASSERT_EQ(firstMediaFile.scalable, true);
  ASSERT_EQ(firstMediaFile.maintainAspectRatio, true);
  ASSERT_EQ(firstMediaFile.resourceUrl, "https://samplemediafileplayback.com/1");

  const auto& secondMediaFile = mediaFiles[1];
  ASSERT_EQ(secondMediaFile.identifier, "mediafile2");
  ASSERT_EQ(secondMediaFile.delivery, "progressive");
  ASSERT_EQ(secondMediaFile.width, 320U);
  ASSERT_EQ(secondMediaFile.height, 240U);
  ASSERT_EQ(secondMediaFile.type, "video/mp4");
  ASSERT_EQ(secondMediaFile.bitrate, 200U);
  ASSERT_EQ(secondMediaFile.scalable, false);
  ASSERT_EQ(secondMediaFile.maintainAspectRatio, false);
  ASSERT_EQ(secondMediaFile.resourceUrl, "https://samplemediafileplayback.com/2");
}

TEST_F(AdsApiTest, GetInlineAd) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                    .StrictRequestParams(false)
                    .SetResponseBodyFromFile("ads/sample_inline_ad.xml")
                    .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(config, requestInfo, [&callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& ads) {
    EXPECT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(ads.size(), 1U);
    const auto& ad = ads[0];

    VerifyBasicInfoFromInlineLinearSample(ad);

    ASSERT_EQ(ad.errorUrls.size(), 1U);
    ASSERT_EQ(ad.errorUrls[0], "https://sampleerrorurl.com?error=[ERRORCODE]");
    ASSERT_EQ(ad.impressions.size(), 1U);

    const auto& impression = ad.impressions[0];
    ASSERT_EQ(impression.identifier, "impressionid");
    ASSERT_EQ(impression.resourceUrl, "https://sampleimpressionurl.com");

    ASSERT_EQ(ad.linearCreatives.size(), 1U);
    const auto& linear = ad.linearCreatives[0];

    ASSERT_EQ(linear.trackingEventUrls.size(), 2U);

    const auto& startTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Start)->second;
    ASSERT_EQ(startTrackingEvents.size(), 1U);
    ASSERT_EQ(startTrackingEvents[0], "https://sampletrackingstart.com");

    const auto& completeTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Complete)->second;
    ASSERT_EQ(completeTrackingEvents.size(), 1U);
    ASSERT_EQ(completeTrackingEvents[0], "https://sampletrackingcomplete.com");

    const auto& clickTrackingUrls = linear.clickTrackingUrls;
    ASSERT_EQ(clickTrackingUrls.size(), 2U);
    ASSERT_EQ(clickTrackingUrls[0], "https://sampleclicktracking.com/1");
    ASSERT_EQ(clickTrackingUrls[1], "https://sampleclicktracking.com/2");

    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response);
}

TEST_F(AdsApiTest, GetAdFailure) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                    .StrictRequestParams(false)
                    .SetStatusCode(404)
                    .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(config, requestInfo, [&callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& /*ads*/) {
    EXPECT_FALSE(TTV_SUCCEEDED(ec));

    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response);
}

TEST_F(AdsApiTest, GetWrapperSuccess) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response1 = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_wrapper_ad.xml")
                     .Done();

  auto response2 = mHttpRequest->AddResponse("https://samplewrappertagurl.com")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_inline_ad.xml")
                     .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(config, requestInfo, [&callbackReceived](TTV_ErrorCode /*ec*/, std::vector<Ad>&& ads) {
    ASSERT_EQ(ads.size(), 1U);
    const auto& ad = ads[0];

    // Ensure the ad inherited some of the values from the wrapper.
    ASSERT_EQ(ad.errorUrls.size(), 2U);
    ASSERT_NE(std::find(ad.errorUrls.begin(), ad.errorUrls.end(), "https://sampleerrorurl.com?error=[ERRORCODE]"),
      ad.errorUrls.end());
    ASSERT_NE(
      std::find(ad.errorUrls.begin(), ad.errorUrls.end(), "https://samplewrappererrorurl.com?error=[ERRORCODE]"),
      ad.errorUrls.end());

    ASSERT_EQ(ad.impressions.size(), 2U);
    ASSERT_NE(std::find_if(ad.impressions.begin(), ad.impressions.end(),
                [](const ads::Impression& impression) {
                  return (impression.identifier == "impressionid") &&
                         (impression.resourceUrl == "https://sampleimpressionurl.com");
                }),
      ad.impressions.end());
    ASSERT_NE(std::find_if(ad.impressions.begin(), ad.impressions.end(),
                [](const ads::Impression& impression) {
                  return (impression.identifier == "wrapperimpressionid") &&
                         (impression.resourceUrl == "https://samplewrapperimpression.com");
                }),
      ad.impressions.end());

    ASSERT_EQ(ad.linearCreatives.size(), 1U);
    const auto& linear = ad.linearCreatives[0];

    ASSERT_EQ(linear.trackingEventUrls.size(), 3U);

    const auto& startTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Start)->second;
    ASSERT_EQ(startTrackingEvents.size(), 2U);
    ASSERT_NE(std::find(startTrackingEvents.begin(), startTrackingEvents.end(), "https://sampletrackingstart.com"),
      startTrackingEvents.end());
    ASSERT_NE(
      std::find(startTrackingEvents.begin(), startTrackingEvents.end(), "https://samplewrappertrackingstart.com"),
      startTrackingEvents.end());

    const auto& completeTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Complete)->second;
    ASSERT_EQ(completeTrackingEvents.size(), 1U);
    ASSERT_EQ(completeTrackingEvents[0], "https://sampletrackingcomplete.com");

    const auto& closeTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Close)->second;
    ASSERT_EQ(closeTrackingEvents.size(), 1U);
    ASSERT_EQ(closeTrackingEvents[0], "https://samplewrappertrackingclose.com");

    const auto& clickTrackingUrls = linear.clickTrackingUrls;
    ASSERT_EQ(clickTrackingUrls.size(), 4U);
    ASSERT_NE(std::find(clickTrackingUrls.begin(), clickTrackingUrls.end(), "https://sampleclicktracking.com/1"),
      clickTrackingUrls.end());
    ASSERT_NE(std::find(clickTrackingUrls.begin(), clickTrackingUrls.end(), "https://sampleclicktracking.com/2"),
      clickTrackingUrls.end());
    ASSERT_NE(std::find(clickTrackingUrls.begin(), clickTrackingUrls.end(), "https://samplewrapperclicktracking.com/1"),
      clickTrackingUrls.end());
    ASSERT_NE(std::find(clickTrackingUrls.begin(), clickTrackingUrls.end(), "https://samplewrapperclicktracking.com/2"),
      clickTrackingUrls.end());

    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
}

TEST_F(AdsApiTest, GetWrapperFailure) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response1 = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_wrapper_ad.xml")
                     .Done();

  auto response2 =
    mHttpRequest->AddResponse("https://samplewrappertagurl.com").StrictRequestParams(false).SetStatusCode(404).Done();

  bool receivedRequests = false;
  auto response3 =
    mHttpRequest->AddResponse("https://samplewrappererrorurl.com")
      .StrictRequestParams(false)
      .SetResponseHandler([&receivedRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        receivedRequests = true;
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 401);
      })
      .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(
    config, requestInfo, [&receivedRequests, &callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& /*ads*/) {
      EXPECT_TRUE(receivedRequests);
      EXPECT_TRUE(TTV_SUCCEEDED(ec));

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
  mHttpRequest->RemoveResponse(response3);
}

TEST_F(AdsApiTest, AdWrapperTooManyRedirects) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response1 = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_wrapper_ad.xml")
                     .Done();

  auto response2 = mHttpRequest->AddResponse("https://samplewrappertagurl.com")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_wrapper_ad.xml")
                     .Done();

  int errorRequests = 0;
  auto response3 =
    mHttpRequest->AddResponse("https://samplewrappererrorurl.com?error=302")
      .StrictRequestParams(true)
      .SetResponseHandler([&errorRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        errorRequests++;
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 401);
      })
      .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(
    config, requestInfo, [&errorRequests, &callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& /*ads*/) {
      EXPECT_EQ(errorRequests, 4);
      EXPECT_TRUE(TTV_SUCCEEDED(ec));

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
  mHttpRequest->RemoveResponse(response3);
}

TEST_F(AdsApiTest, AdWrapperNoInlineAd) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSamplePrerollAdFetchRequest(requestInfo);

  auto response1 = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_wrapper_ad.xml")
                     .Done();

  auto response2 = mHttpRequest->AddResponse("https://samplewrappertagurl.com")
                     .StrictRequestParams(false)
                     .SetResponseBodyFromFile("ads/sample_no_ad.xml")
                     .Done();

  bool receivedRequests = false;
  auto response3 =
    mHttpRequest->AddResponse("https://samplewrappererrorurl.com?error=303")
      .StrictRequestParams(true)
      .SetResponseHandler([&receivedRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        receivedRequests = true;
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 401);
      })
      .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(
    config, requestInfo, [&receivedRequests, &callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& /*ads*/) {
      EXPECT_TRUE(receivedRequests);
      EXPECT_TRUE(TTV_SUCCEEDED(ec));

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
  mHttpRequest->RemoveResponse(response3);
}

TEST_F(AdsApiTest, AdReport) {
  AdConfiguration config;
  SetAdConfiguration(config);

  auto reportUrls = std::vector<std::string>{"https://reporturl.com/[MACRO1]?macro2=[MACRO2]",
    "https://reporturl.com/[MACRO2]", "https://reporturl.com/[UNSUBSTITUTED]"};

  auto macroSubsitutions = std::map<std::string, std::string>{{"macro1", "sub1"}, {"macro2", "sub2"}};

  size_t numRequests = 0;

  auto response1 =
    mHttpRequest->AddResponse("https://reporturl.com/sub1?macro2=sub2")
      .StrictRequestParams(true)
      .SetResponseHandler([&numRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        numRequests++;
        return MockResponse::MockResponseContent("", std::map<std::string, std::string>(), 200);
      })
      .Done();

  auto response2 =
    mHttpRequest->AddResponse("https://reporturl.com/sub2")
      .StrictRequestParams(true)
      .SetResponseHandler([&numRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        numRequests++;
        return MockResponse::MockResponseContent(
          "", std::map<std::string, std::string>(), (numRequests == 2) ? 200 : 401);
      })
      .Done();

  auto response3 =
    mHttpRequest->AddResponse("https://reporturl.com/")
      .StrictRequestParams(true)
      .SetResponseHandler([&numRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
        numRequests++;
        return MockResponse::MockResponseContent(
          "", std::map<std::string, std::string>(), (numRequests == 3) ? 200 : 401);
      })
      .Done();

  size_t numCallbacksReceived = 0;

  for (const auto& url : reportUrls) {
    auto ec =
      mAdsApi->ReportAdEvent(url, macroSubsitutions, "Test User Agent", [&numCallbacksReceived](TTV_ErrorCode ec) {
        ASSERT_TRUE(TTV_SUCCEEDED(ec));
        numCallbacksReceived++;
      });
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  }

  size_t reportUrlsSize = reportUrls.size();
  WaitUntilResult(kTestTimeoutMilliseconds, [&numCallbacksReceived, &numRequests, &reportUrlsSize]() {
    return (numCallbacksReceived == reportUrlsSize) && (numRequests == reportUrlsSize);
  });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
  mHttpRequest->RemoveResponse(response3);
}

TEST_F(AdsApiTest, GetPlaylistAd) {
  AdConfiguration config;
  SetAdConfiguration(config);

  AdFetchRequestInfo requestInfo;
  SetSampleMidrollAdFetchRequest(requestInfo);

  std::string redirectedString = "https://redirectedadurl.com";

  bool receivedRequests = false;
  auto response1 = mHttpRequest->AddResponse("https://pubads.g.doubleclick.net/gampad/ads")
                     .StrictRequestParams(false)
                     .SetResponseHandler(
                       [&redirectedString, &receivedRequests](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                         const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& /*body*/) {
                         receivedRequests = true;
                         return MockResponse::MockResponseContent(
                           "<Playlist><Preroll><Ad>" + redirectedString + "</Ad></Preroll></Playlist>",
                           std::map<std::string, std::string>(), 200);
                       })
                     .Done();

  auto response2 = mHttpRequest->AddResponse(redirectedString)
                     .StrictRequestParams(true)
                     .SetResponseBodyFromFile("ads/sample_inline_ad.xml")
                     .Done();

  bool callbackReceived = false;
  auto ec = mAdsApi->FetchAds(
    config, requestInfo, [&receivedRequests, &callbackReceived](TTV_ErrorCode ec, std::vector<Ad>&& ads) {
      ASSERT_EQ(ads.size(), 1U);
      const auto& ad = ads[0];

      VerifyBasicInfoFromInlineLinearSample(ad);

      ASSERT_EQ(ad.errorUrls.size(), 1U);
      ASSERT_EQ(ad.errorUrls[0], "https://sampleerrorurl.com?error=[ERRORCODE]");
      ASSERT_EQ(ad.impressions.size(), 1U);

      const auto& impression = ad.impressions[0];
      ASSERT_EQ(impression.identifier, "impressionid");
      ASSERT_EQ(impression.resourceUrl, "https://sampleimpressionurl.com");

      ASSERT_EQ(ad.linearCreatives.size(), 1U);
      const auto& linear = ad.linearCreatives[0];

      ASSERT_EQ(linear.trackingEventUrls.size(), 2U);

      const auto& startTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Start)->second;
      ASSERT_EQ(startTrackingEvents.size(), 1U);
      ASSERT_EQ(startTrackingEvents[0], "https://sampletrackingstart.com");

      const auto& completeTrackingEvents = linear.trackingEventUrls.find(ads::TrackingEventType::Complete)->second;
      ASSERT_EQ(completeTrackingEvents.size(), 1U);
      ASSERT_EQ(completeTrackingEvents[0], "https://sampletrackingcomplete.com");

      const auto& clickTrackingUrls = linear.clickTrackingUrls;
      ASSERT_EQ(clickTrackingUrls.size(), 2U);
      ASSERT_EQ(clickTrackingUrls[0], "https://sampleclicktracking.com/1");
      ASSERT_EQ(clickTrackingUrls[1], "https://sampleclicktracking.com/2");

      EXPECT_TRUE(receivedRequests);
      EXPECT_TRUE(TTV_SUCCEEDED(ec));

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(kTestTimeoutMilliseconds, [&callbackReceived]() { return callbackReceived; });

  mHttpRequest->RemoveResponse(response1);
  mHttpRequest->RemoveResponse(response2);
}
