/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/module.h"

#include <unordered_set>

#include <atomic>
#include <map>
#include <memory>

namespace ttv {
class TaskRunner;
class UserRepository;
class SettingRepository;

namespace broadcast {
class BroadcastAPI;
class BroadcastApiInternalData;
class IBroadcastAPIListener;
class IIngestTesterListener;
class UserBroadcastStateHost;
class TwitchAPI;
class Streamer;
class IVideoEncoder;
class IAudioEncoder;
class IAudioCapture;
class IVideoCapture;
class IIngestTester;
class IBandwidthStatListener;
class IMuxer;
}  // namespace broadcast
}  // namespace ttv

/**
 * This module provides the ability to broadcast to Twitch by accepting video and audio data from
 * a game/app, encoding it and sending it to the Twitch backend in realtime.
 *
 * The following properties must be set before calling Initialize().
 * - SetCoreApi()
 * - SetListener()
 *
 * In order to broadcast several services must be configured using the following methods.
 * - SetVideoCapturer(): Sets the component that will supply video frames for encoding.
 * - SetVideoEncoder(): Sets the H.264 video encoder.  This will be platform specific.
 * - SetAudioCapturer(): Sets the components that will supply audio for encoding.  This
 *   may be called multiple times, such as for microphone and system audio.
 * - SetAudioEncoder(): Sets the audio encoder.  This will be platform specific.
 *
 * The video parameters must also be set via one of the 2 following methods.
 * - SetVideoParams(): Sets parameters when you know your maximum bandwidth.
 * - SetAdvancedVideoParams(): Sets parameters when you don't care about bandwidth and want
 *   to ensure you have a specific resolution.  This is not recommended for live streams.
 *
 * Currently, only one user may be broadcasting at a time.  In order to specify which user will
 * broadcast SetActiveUser() must be used.  The active user must be set and the user logged in
 * before configuring any of the encoders, capturers and other settings.  If the active user is
 * changed or logged out then the configuration is lost.
 *
 * If you have a custom application which has a need to access the raw encoded data then
 * see the SetCustomMuxer() method.
 */
class ttv::broadcast::BroadcastAPI : public ttv::ModuleBase {
 public:
  using StartBroadcastCallback = std::function<void(TTV_ErrorCode ec)>;
  using StopBroadcastCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchIngestListCallback = std::function<void(TTV_ErrorCode ec, std::vector<IngestServer>&& result)>;
  using RunCommercialCallback = std::function<void(TTV_ErrorCode ec)>;
  using SetStreamInfoCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  BroadcastAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  /**
   * Sets the CoreAPI instance to use for low level services.  This must be set before
   * Initialize() is called.
   */
  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);
  /**
   * Sets the client listener implementation.  This must be set before Initialize() is called.
   */
  TTV_ErrorCode SetListener(const std::shared_ptr<IBroadcastAPIListener>& listener);
  /**
   * Sets an optional receiver of post-encoded video and audio data.
   * This can only be called before starting a broadcast.
   * @param muxer A custom IMuxer implementation which allow an application to obtain the
   * raw encoded data.
   */
  TTV_ErrorCode SetCustomMuxer(const std::shared_ptr<IMuxer>& muxer);
  /**
   * Sets the broadcaster value associated with the stream so the software used can be identified.
   */
  TTV_ErrorCode SetBroadcasterSoftware(const std::string& str);
  /**
   * Sets the flag which forces the recording of the broadcast on the server (recorder=1 in the RTMP URL)
   */
  TTV_ErrorCode SetForceArchiveBroadcast(bool forceArchive);
  /**
   * Sets the flag which forces not to record the broadcast on the server (recorder=0 in the RTMP URL)
   */
  TTV_ErrorCode SetForceDontArchiveBroadcast(bool forceDontArchive);
  /**
   * Sets the userid of the user who will be broadcasting.  This must be set before attempting to
   * broadcast and cannot be changed while broadcasting.  See the class
   * description for more details.
   */
  TTV_ErrorCode SetActiveUser(UserId userId);
  /**
   * Sets the desired video encoder to use.
   */
  TTV_ErrorCode SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder);
  /**
   * Sets the desired audio encoder to use.
   */
  TTV_ErrorCode SetAudioEncoder(const std::shared_ptr<IAudioEncoder>& encoder);
  /**
   * Sets the source of video frames.
   */
  TTV_ErrorCode SetVideoCapturer(const std::shared_ptr<IVideoCapture>& capturer);
  /**
   * Sets the source of audio samples for the given layer.  The layer ids are up to the client but
   * generally start at 0 and increment for each layer the application adds.  For instance, a
   * capturer for system audio could be added for layer 0 and a microphone capturer added for
   * layer 1.
   */
  TTV_ErrorCode SetAudioCapturer(AudioLayerId layer, const std::shared_ptr<IAudioCapture>& capturer);
  /**
   * Removes a previously added audio capturer for the given layer.
   */
  TTV_ErrorCode RemoveAudioCapturer(AudioLayerId layer);
  /**
   * Sets the volume of given audio layer and can be toggled freely during a stream.
   * The IAudioCapture for the layer must already be set via SetAudioCapturer().
   */
  TTV_ErrorCode SetAudioLayerVolume(AudioLayerId layer, float volume);
  /**
   * This mutes the audio for the given audio layer and can be toggled freely during a stream.
   * The IAudioCapture for the layer must already be set via SetAudioCapturer().
   */
  TTV_ErrorCode SetAudioLayerMuted(AudioLayerId layer, bool muted);
  /**
   * This will enable or disable the associated capturer for the entire broadcast.  This cannot be
   * changed while broadcasting.
   * The IAudioCapture for the layer must already be set via SetAudioCapturer().
   */
  TTV_ErrorCode SetAudioLayerEnabled(AudioLayerId layer, bool enabled);
  /**
   * Sets the VideoParams to use for the broadcast.  This cannot be set while a stream is in progress.
   */
  TTV_ErrorCode SetVideoParams(const VideoParams& videoParams);
  /**
   * Retrieves the currently set VideoParams.
   */
  TTV_ErrorCode GetVideoParams(VideoParams& videoParams);
  /**
   * Sets the output FLV file path for saving a broadcast locally for debugging and verification.
   */
  TTV_ErrorCode SetOutputPath(const std::wstring& outputPath);
  /**
   * Sets the type of connection that will be used to broadcast. This value is used when sending tracking events.
   */
  TTV_ErrorCode SetConnectionType(ConnectionType connectionType);
  /**
   * Sets a unique ID for the broadcasting session, which can span multiple starts/stops. This value is used when
   * sending tracking events.
   */
  TTV_ErrorCode SetSessionId(const std::string& sessionId);
  /**
   * Adds a listener that will be periodically called back with statistics about network usage. This should not be
   * invoked while broadcasting.
   */
  TTV_ErrorCode AddBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener);
  /**
   * Removes a previously added bandwidth stat listener. This should not be invoked while broadcasting.
   */
  TTV_ErrorCode RemoveBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener);
  /**
   * Initiates broadcasting for the active user.
   */
  TTV_ErrorCode StartBroadcast(StartBroadcastCallback&& callback);
  /**
   * Gracefully stops broadcasting.
   * @param[in] reason A string describing why the stream is being stopped, used for tracking. Pass
   * in "user_ended" if this was an explicit action from the user.
   */
  TTV_ErrorCode StopBroadcast(const std::string& reason, StopBroadcastCallback&& callback);
  /**
   * Retrieves the current time in the live broadcast.
   */
  TTV_ErrorCode GetCurrentBroadcastTime(uint64_t& broadcastTime);
  /**
   * Fetches the list of available ingest servers that can be used for broadcasting.
   */
  TTV_ErrorCode FetchIngestServerList(FetchIngestListCallback&& callback);
  /**
   * Retrieves the currently selected ingest server.  It's possible that none is selected and will
   * contain empty values.
   */
  TTV_ErrorCode GetSelectedIngestServer(IngestServer& result);
  /**
   * Sets the selected ingest server to use when initiating broadcasts.
   */
  TTV_ErrorCode SetSelectedIngestServer(const IngestServer& server);

  /**
   * Retrieves the current state of broadcasting.
   */
  TTV_ErrorCode GetBroadcastState(BroadcastState& result) const;

  /**
   * Enable or disable async FlvMuxer. It queues up frames while connecting to the outputs (rtmp).
   */
  TTV_ErrorCode SetFlvMuxerAsyncEnabled(bool enable);
  TTV_ErrorCode GetFlvMuxerAsyncEnabled(bool& isEnabledResult);

  /**
   * Triggers a commercial of the specified duration in seconds.  This may be triggered
   * independently from whether or not a broadcast is in progress.  Note
   * that only partnered users may run commercials.
   */
  TTV_ErrorCode RunCommercial(
    UserId userId, ChannelId channelId, uint32_t timebreakSeconds, RunCommercialCallback&& callback);
  /**
   * Sets the current game and stream title for a channel.  These values may be set independently
   * from whether or not a broadcast is in progress.
   */
  TTV_ErrorCode SetStreamInfo(UserId userId, ChannelId channelId, const std::string& game, const std::string& title,
    SetStreamInfoCallback&& callback);
  /**
   * Returns an implementation of IIngestTester for the given owning user.  When done with the
   * implementation it must be released via DisposeIngestTester().
   */
  TTV_ErrorCode CreateIngestTester(UserId userId, const std::shared_ptr<IIngestTesterListener>& listener,
    const uint8_t* testDataBuffer, uint32_t testDataLength, std::shared_ptr<IIngestTester>& result);
  /**
   * Releases the given IIngestTester implementation that was returned by CreateIngestTester().
   */
  TTV_ErrorCode DisposeIngestTester(const std::shared_ptr<IIngestTester>& ingestTester);
  /**
   * Retrieves the OAuth token scopes required to support all features in this module.
   */
  static void GetRequiredAuthScopes(std::vector<std::string>& scopes);

 private:
  class CoreApiClient : public ICoreApiClient {
   public:
    CoreApiClient(BroadcastAPI* api);

    // ICoreApiClient implementation
    virtual std::string GetClientName();
    virtual void GetRequiredOAuthScopes(std::vector<std::string>& scopes);
    virtual void CoreUserLoggedIn(std::shared_ptr<User> user);
    virtual void CoreUserLoggedOut(std::shared_ptr<User> user);

   private:
    BroadcastAPI* mApi;
  };

  // ICoreApiClient implementation
  void CoreUserLoggedIn(std::shared_ptr<User> user);
  void CoreUserLoggedOut(std::shared_ptr<User> user);

  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  std::shared_ptr<Streamer> CreateStreamer(const std::shared_ptr<User>& user);
  std::shared_ptr<TwitchAPI> CreateTwitchAPI(const std::shared_ptr<User>& user);
  TTV_ErrorCode GetStreamer(std::shared_ptr<Streamer>& result);
  TTV_ErrorCode EnsureBroadcastingAndGetStreamer(std::shared_ptr<Streamer>& result);
  TTV_ErrorCode EnsureNotBroadcastingAndGetStreamer(std::shared_ptr<Streamer>& result);

  TTV_ErrorCode BindToUser(const std::shared_ptr<User>& user);

  void SetBroadcastState(TTV_ErrorCode ec, BroadcastState state);

  /**
   * Releases the given IIngestTester implementation that was returned by CreateIngestTester().
   * Does not require an instance of the API to be alive to use it.
   */
  static TTV_ErrorCode DisposeIngestTesterInternal(
    const std::shared_ptr<IIngestTester>& ingestTester, const std::shared_ptr<BroadcastApiInternalData>& internalData);

  std::shared_ptr<BroadcastApiInternalData> mInternalData;
  std::shared_ptr<CoreApiClient> mCoreApiClient;  //!< The client instance registered with CoreAPI.
};
