/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/types/coretypes.h"

namespace ttv {
namespace broadcast {
class IBroadcastAPIListener;
class IIngestTesterListener;
class IngestTesterListenerProxy;
class IIngestTester;
class IBandwidthStatListener;
}  // namespace broadcast
}  // namespace ttv

/**
 * Listener interface for callbacks from BroadcastAPI.
 */
class ttv::broadcast::IBroadcastAPIListener : public ttv::IModuleListener {
 public:
  /**
   * Fired when the state of the broadcast changes for any reason.
   * If the BroadcastState is Stopping, we will return TTV_EC_SUCCESS if it was a solicited stop (from a call
   * to StopBroadcast). Otherwise it will reflect the underlying error that caused the broadcast to end.
   * Some examples include:
   *   TTV_EC_SOCKET_ENOTCONN, TTV_EC_SOCKET_ECONNABORTED, TTV_EC_SOCKET_SEND_ERROR, TTV_EC_SOCKET_RECV_ERROR
   *     Socket failures when communicating with the server.
   *   TTV_EC_BROADCAST_RTMP_TIMEOUT
   *     Server stopped responding to our RTMP data. This can happen if there is another broadcasting session running.
   *   TTV_EC_HTTPREQUEST_ERROR
   *     HTTP failure when attempting to fetch some of the channel data for the stream.
   */
  virtual void BroadcastStateChanged(TTV_ErrorCode ec, BroadcastState state) = 0;
  /**
   * Fired periodically when there are issues with the user's bandwidth while broadcasting.  If the connection
   * backs up too much then a proper error wil be issues and the broadcast will automatically terminate.  If/when
   * issues have recovered then TTV_EC_SUCCESS will be fired once to clear the warning state.
   *
   * @param ec The severity of the warning.  Possible values are TTV_EC_BROADCAST_FRAMES_QUEUEING,
   *   TTV_EC_BROADCAST_FRAME_QUEUE_TOO_LONG and TTV_EC_SUCCESS.
   * @param backupMilliseconds The number of milliseconds which are backed up on the connection.
   */
  virtual void BroadcastBandwidthWarning(TTV_ErrorCode ec, uint32_t backupMilliseconds) = 0;
  virtual void BroadcastFrameSubmissionIssue(TTV_ErrorCode ec) = 0;
  /**
   * Fired when we get the broadcast_id from the livecheck API. The broadcast_id (also known as stream_id) is
   * the id of the stream that the user is current broadcasting. This should fire 5-20s after the broadcast starts
   * once we get the broadcast_id from the livecheck API, but could take a while due to latency (10s~) on the endpoint.
   *
   * The callback may not be called if the broadcast stops before we have a chance to fetch it.
   */
  virtual void StreamInfoFetched(TTV_ErrorCode ec, const StreamInfo& streamInfo) = 0;
  /**
   * Fired when a StreamKey request returns a CanThey Error.
   */
  virtual void StreamKeyError(const CanTheyError& error) = 0;
};

/**
 * Listener interface for callbacks from IngestTester.
 */
class ttv::broadcast::IIngestTesterListener {
 public:
  virtual ~IIngestTesterListener() = default;

  /**
   * Fired when the state of the tester has changed.
   */
  virtual void BroadcastIngestTesterStateChanged(IIngestTester* source) = 0;
};

/**
 * A lambda proxy for IIngestTesterListener.
 */
class ttv::broadcast::IngestTesterListenerProxy : public ttv::broadcast::IIngestTesterListener {
 public:
  using OnStateChangedFunc = std::function<void(IIngestTester* source)>;

 public:
  // IIngestTesterListener implementation
  virtual void BroadcastIngestTesterStateChanged(IIngestTester* source) override {
    if (mOnStateChangedFunc != nullptr) {
      mOnStateChangedFunc(source);
    }
  }

 public:
  OnStateChangedFunc mOnStateChangedFunc;
};

/**
 * Listener interface that gets called back periodically with statistics about the current stream.
 */
class ttv::broadcast::IBandwidthStatListener {
 public:
  virtual ~IBandwidthStatListener() = default;
  virtual void ReceivedBandwidthStat(const BandwidthStat& stat) = 0;
};
