/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasterrortypes.h"
#include "twitchsdk/core/types/coretypes.h"

#include <vector>

namespace ttv {
namespace broadcast {
const uint32_t kMinBitRate = 300;              //!< The minimum bit rate supported by the Twitch backend.
const uint32_t kMaxBitRate = 6000;             //!< The absolute maximum bit rate supported by the Twitch backend.
const uint32_t kDefaultInitialBitRate = 1500;  //!< The default starting bit rate of a broadcast.
const uint32_t kDefaultFramesPerSecond = 30;   //!< The default frames per second of the broadcast.
const uint32_t kMinFramesPerSecond = 10;       //!< The minimum frames per second supported by the Twitch backend.
const uint32_t kMaxFramesPerSecond = 60;       //!< The maximum frames per second supported by the Twitch backend.
const uint32_t kMaxFrameWidth = 1920;  //!< The maximum width supported by the Twitch backend. Must be a multiple of 16.
const uint32_t kMaxFrameHeight =
  1200;  //!< The maximum height supported by the Twitch backend. Must be a multiple of 16.
const float kRecommendedBitsPerPixel = 0.1f;

/**
 * Represents a Twitch Ingest server.
 */
struct IngestServer {
  IngestServer();
  IngestServer(const IngestServer& other);
  void operator=(const IngestServer& other);

  std::string serverName;  //!< The name of the server suitable for displaying in UI.
  std::string serverUrl;   //!< The URL of the Twitch server.
  uint32_t priority;       //!< The priority of the server, the lower the better option for the client.
  uint32_t serverId;       //!< The unique server ID which should not change.
};

/**
 * Supported input frame pixel formats.
 */
enum class PixelFormat {
  TTV_PF_BGRA = 0x00010203,
  TTV_PF_ABGR = 0x01020300,
  TTV_PF_RGBA = 0x02010003,
  TTV_PF_ARGB = 0x03020100
};

/**
 * Supported YUV convertion types.
 */
enum class YUVFormat {
  TTV_YUV_NONE = -1,
  TTV_YUV_I420,  //!< 8 bit Y plane followed by 8 bit 2x2 subsampled U and V planes (12 bits/pixel)
  TTV_YUV_YV12,  //!< 8 bit Y plane followed by 8 bit 2x2 subsampled V and U planes (12 bits/pixel)
  TTV_YUV_NV12   //!< 8-bit Y plane followed by an interleaved U/V plane with 2x2 subsampling (12 bits/pixel)
};

/**
 * Set CPU usage level for video encoding.  This may not have any effect on some platforms and encoders.
 */
enum class EncodingCpuUsage { Default, Low, Medium, High };

/**
 * Video parameters that are set at the beginning of a broadcast session.
 */
struct VideoParams {
  VideoParams();

  /**
   * The width of the output video stream in pixels.  The IVideoCapturer should produce frames with
   * this exact width for best quality.
   */
  uint32_t outputWidth;
  /**
   * The height of the output video stream in pixels.  The IVideoCapturer should produce frames with
   * this exact height for best quality.
   */
  uint32_t outputHeight;
  /**
   * The number of frames per second that will be submitted by the capturer and output in the stream.
   */
  uint32_t targetFramesPerSecond;
  /**
   * The starting bit rate in 1000s of bits per second.  If automatic bit rate adjustment is disabled then
   * this will be the constant bit rate used by the video encoder.
   */
  uint32_t initialKbps;
  /**
   * When automatic bit rate adjustment is enabled, this is minimum bit rate in 1000s of bits per second.
   */
  uint32_t minimumKbps;
  /**
   * When automatic bit rate adjustment is enabled, this is maximum bit rate in 1000s of bits per second.
   */
  uint32_t maximumKbps;
  /**
   * The desired CPU usage level for video encoding.  This property may not have any effect on some encoders and
   * platforms.
   */
  EncodingCpuUsage encodingCpuUsage;
  /**
   * Whether or not to adjust bit rate dynamically based on available bandwidth.
   */
  bool automaticBitRateAdjustmentEnabled;

  /**
   * Configures the video parameters so that a broadcast stays within the given maxKbps.  The
   * resolution of the broadcast will be computed automatically based on given parameters.
   *
   * @param[in] expectedKbps The expeected bit rate of the connection in 1000s of bits per second.
   * @param[in] framesPerSecond The number of frames per second that will be submitted by the capturer and output in the
   * stream.
   * @param[in] bitsPerPixel The number of bits to use per pixel.  A good recommended value is 0.1.
   * @param[in] aspectRatio The desired aspect ratio of the calculated resolution.
   * @return
   *   - TTV_EC_SUCCESS - The configuration was successful.
   *   - TTV_EC_BROADCAST_INVALID_BITRATE - The bit rate was not within the required range for the Twitch backend.
   *   - TTV_EC_BROADCAST_INVALID_FPS - The frames per second value was not within the required range for the Twitch
   * backend.
   *   - TTV_EC_INVALID_ARG - The aspect ratio or bits per pixel were not valid inputs.
   */
  static TTV_ErrorCode ConfigureForBandwidth(
    uint32_t expectedKbps, uint32_t framesPerSecond, float bitsPerPixel, float aspectRatio, VideoParams& result);
  /**
   * Configures the broadcast settings directly allowing a specific resolution and framerate to be
   * used.  Please note that this method may result in a broadcast that tries to send more data
   * than a user's connection can support.  Use SetVideoParams() along with ingest testing to
   * ensure the best stream quality available.
   *
   * @param[in] width The desired width of the stream in pixels.
   * @param[in] height The desired height of the stream in pixels.
   * @param[in] framesPerSecond The number of frames per second that will be submitted by the capturer and output in the
   * stream.
   * @param[in] bitsPerPixel The number of bits to use per pixel.  A good recommended value is 0.1.
   * @return
   *   - TTV_EC_SUCCESS - The configuration was successful.
   *   - TTV_EC_BROADCAST_INVALID_RESOLUTION - The resolution was not within the required range for the Twitch backend.
   *   - TTV_EC_BROADCAST_INVALID_FPS - The frames per second value was not within the required range for the Twitch
   * backend.
   */
  static TTV_ErrorCode ConfigureForResolution(
    uint32_t width, uint32_t height, uint32_t framesPerSecond, float bitsPerPixel, VideoParams& result);
};

enum class ConnectionType { Wifi, Ethernet, Cellular, Unknown };

/**
 * The data format of audio samples.
 */
enum class AudioFormat {
  None,  //!< No audio
  PCM,   //!< Pulse-code modulation
  MP3,   //!< MPEG-2 Audio Layer III
  AAC,   //!< Advanced Audio Coding
};

/**
 * Supported input audio sample formats
 */
enum class AudioSampleFormat {
  TTV_ASF_PCM_S16  //!< PCM signed 16-bit
};

/**
 * The supported sample rates in samples per second.
 */
enum class AudioSampleRate { Hz11250 = 11250, Hz22500 = 22500, Hz44100 = 44100 };

using AudioLayerId = uint32_t;

/**
 * The Video recording status of the channel
 */
struct ArchivingState {
  ArchivingState();

  std::string cureUrl;    //!< The URL for where the user can go to enable video recording for the channel
  bool recordingEnabled;  //!< Recording is enabled/disabled for the channel
};

/**
 * Display information about a game.
 */
struct GameInfo {
  GameInfo();

  std::string name;     //!< The display name of the game.
  uint32_t popularity;  //!< A popularity rating for the game.
  uint32_t gameId;      //!< The game's unique id.
};

/**
 * A list of game info structs.
 */
struct GameInfoList {
  std::vector<GameInfo> games;
};

enum class StreamStartFlags { None = 0, BandwidthTest = 1 << 0 };

enum class BroadcastState {
  Initialized,        //!< Initialize has been called.
  ReadyToBroadcast,   //!< Idle and ready to broadcast.
  StartingBroadcast,  //!< Processing a request to start broadcasting.
  Broadcasting,       //!< Currently broadcasting.
  StoppingBroadcast   //!< Processing a request to stop broadcasting.
};

struct BandwidthStat {
  uint64_t
    recommendedBitsPerSecond;  //!< The bit rate the ABS algorithm has recommended to the encoder in bits per second.
  uint64_t
    measuredBitsPerSecond;  //!< The actual measured speed we are sending data over the socket in bits per second.
  uint64_t encoderOutputBitsPerSecond;  //!< The rate at which the vide encoder is actually producing data.
  double backBufferSeconds;             //!< How many seconds of data is buffered to be sent.
  double recordedTime;     //!< The time of the measurement in number of seconds since the start of broadcast.
  double congestionLevel;  //!< What fraction of our time is spent blocking on the network socket, from 0.0-1.0.
};

/**
 * The values required to initialize an video + optional audio stream.
 */
struct MuxerParameters {
  std::string appVersion;  //!< The version of the app using the muxer.

  uint32_t videoWidth;            //!< The width of the video frame.
  uint32_t videoHeight;           //!< The height of the video frame.
  uint32_t frameRate;             //!< The number of frames per second.
  std::vector<uint8_t> videoSps;  //!< Sequence parameter set.
  std::vector<uint8_t> videoPps;  //!< Picture parameter set.

  AudioFormat audioFormat;   //!< The audio encoding format
  uint32_t audioSampleRate;  //!< The number of audio samples per second.
  uint32_t audioSampleSize;  //!< The number of bits in a sample.
  bool audioEnabled;         //!< Whether or not to capture and encode audio.
  bool audioStereo;          //!< Whether or not to encode stereo audio.
};
}  // namespace broadcast
}  // namespace ttv
