/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"

#include <memory>

namespace ttv {
namespace broadcast {
class IAudioMixer;
class IAudioCapture;
}  // namespace broadcast
}  // namespace ttv

/**
 * Interface for capturing and supplying audio.
 *
 * The following methods should be called by the client.
 *   Initialize() should be called by the client prior to adding it to the system.
 *   Shutdown() should be called after removing from the system.
 *
 * The following methods will be called by the system during use.
 *   SetAudioMixer() will be called prior to starting capture.
 *   Start() and Stop() will be called by the system when capture is to be started and stopped.
 */
class ttv::broadcast::IAudioCapture {
 public:
  virtual ~IAudioCapture() = default;

  virtual void SetAudioMixer(const std::shared_ptr<IAudioMixer>& mixer) = 0;
  /**
   * Retrieves the name for logging purposes.
   */
  virtual std::string GetName() const = 0;
  /**
   * Retrieves the number of channels captured.
   */
  virtual uint32_t GetNumChannels() const = 0;
  /**
   * Retrieves the current muted state.
   */
  virtual bool GetMuted() const = 0;
  /**
   * Sets the current muted state.  The capturer should track this flag so it can be returned from GetMuted()
   * but it can also optionally pause capturing when muted if desired.  Silence should still be written
   * as if there were real audio available.
   */
  virtual void SetMuted(bool muted) = 0;
  virtual void SetAudioLayer(AudioLayerId layer) = 0;
  virtual AudioLayerId GetAudioLayer() const = 0;

  virtual TTV_ErrorCode Initialize() = 0;
  virtual TTV_ErrorCode Shutdown() = 0;
  virtual TTV_ErrorCode Start() = 0;
  virtual TTV_ErrorCode Process(const std::shared_ptr<IAudioMixer>& mixer, uint64_t& lastSampleTime) = 0;
  /**
   * This is call once upon starting capture so there is a relative timestamp to work off of.  This
   * is used to pace the capture.
   */
  virtual void SetInitialTime(uint64_t initialTime) = 0;
  virtual TTV_ErrorCode Stop() = 0;
};
