/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/iaudioframereceiver.h"

namespace ttv {
namespace broadcast {
class IFrameWriter;
class IAudioEncoder;
class AudioFrame;
struct AudioParams;
}  // namespace broadcast
}  // namespace ttv

/**
 * Audio parameters
 */
struct ttv::broadcast::AudioParams {
  AudioParams();

  uint32_t numInputAudioLayers;  // The number of audio capturers that will be submitting audio to the mixer.
  uint32_t numChannels;          // The number of audio channels.
  uint32_t encodedSampleRate;    // The samples per second that should come out of the audio encoder.
};

/**
 *
 * Initialize() should be called by the client prior to adding it to the system.
 * Shutdown() should be called after removing from the system.
 * Start() and Stop() will be called by the system when capture is to be started and stopped.
 */
class ttv::broadcast::IAudioEncoder {
 public:
  IAudioEncoder();
  virtual ~IAudioEncoder() = default;

 public:
  virtual TTV_ErrorCode SetFrameWriter(const std::shared_ptr<IFrameWriter>& frameWriter) = 0;
  virtual TTV_ErrorCode GetAudioEncodingFormat(AudioFormat& result) = 0;
  /**
   * Retrieves the name for logging purposes.
   */
  virtual std::string GetName() const = 0;
  virtual TTV_ErrorCode Initialize() = 0;
  virtual TTV_ErrorCode Start(uint32_t streamIndex, const AudioParams& audioParams) = 0;
  /**
   * Assuming the encoder requires a data frame of constant size this is the number of samples in the frame.  If it does
   * not need to be constant then result will be 0.
   */
  virtual TTV_ErrorCode GetNumInputSamplesPerEncodeFrame(uint32_t& result) = 0;
  virtual TTV_ErrorCode Stop() = 0;
  virtual TTV_ErrorCode Shutdown() = 0;

  /**
   * Submits the given audio frame.  It should have been created by using the IAudioFrameReceiver implementation
   * returned by GetReceiverImplementation().
   */
  virtual TTV_ErrorCode SubmitFrame(const std::shared_ptr<AudioFrame>& audioFrame) = 0;
  /**
   * Determines if the concrete implementation supports the given protocol.  The template parameter must be known to
   * derive from IAudioFrameReceiver.
   */
  template <typename T>
  bool SupportsReceiverProtocol() const {
    return SupportsReceiverProtocol(T::GetReceiverTypeId());
  }
  /**
   * Determines if the concrete implementation supports the given protocol.
   */
  virtual bool SupportsReceiverProtocol(IAudioFrameReceiver::ReceiverTypeId typeId) const = 0;
  /**
   * Obtains the receiver for the given protocol.  If not implemented, null will be returned.
   */
  virtual std::shared_ptr<IAudioFrameReceiver> GetReceiverImplementation(
    IAudioFrameReceiver::ReceiverTypeId typeId) = 0;
};
