/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/core/component.h"

#include <functional>
#include <memory>

namespace ttv {
namespace broadcast {
class IIngestTester;
class IVideoEncoder;
}  // namespace broadcast
}  // namespace ttv

/**
 * The interface for the ingest testing service.  This will test a single ingest server.
 */
class ttv::broadcast::IIngestTester {
 public:
  enum class TestState {
    Stopped,        // Start has not yet been called
    Connecting,     // Connecting to the ingest server
    Testing,        // Testing the server
    Disconnecting,  // Disconnecting from the server
    Finished,       // Successfully finished the testing
    Failed          // The test failed for some reason and has stopped
  };

 public:
  virtual ~IIngestTester();
  /**
   * Retrieves the current state of the test.
   */
  virtual TTV_ErrorCode GetTestState(TestState& result) const = 0;
  /**
   * Returns the ingest server currently being tested.  If there is no server then TTV_EC_INVALID_STATE will be
   * returned.
   */
  virtual TTV_ErrorCode GetIngestServer(IngestServer& result) const = 0;
  /**
   * Start the testing the given server.
   */
  virtual TTV_ErrorCode Start(const IngestServer& ingestServer) = 0;
  /**
   * Stop testing the current server and abort the rest of the test.  This will cause the test to fail with error
   * TTV_EC_REQUEST_ABORTED.
   */
  virtual TTV_ErrorCode Cancel() = 0;
  /**
   * Retrieves the computed kilobits (1000 bits) per second of the connection to the ingest server.
   */
  virtual TTV_ErrorCode GetMeasuredKbps(uint32_t& result) = 0;
  /**
   * Retrives the error associated with the test, if available.
   */
  virtual TTV_ErrorCode GetTestError(TTV_ErrorCode& result) = 0;
  /**
   * Sets the number of milliseconds in which to test each server.
   */
  virtual TTV_ErrorCode SetTestDurationMilliseconds(uint64_t duration) = 0;
  /**
   * Gets the number of milliseconds in which to test each server.
   */
  virtual TTV_ErrorCode GetTestDurationMilliseconds(uint64_t& result) const = 0;
  /**
   * The overall progress between [0,1].
   */
  virtual TTV_ErrorCode GetProgress(float& result) const = 0;
  /**
   * The owning user id.
   */
  virtual TTV_ErrorCode GetUserId(UserId& result) const = 0;
};
