/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/packet.h"

namespace ttv {
namespace broadcast {
class IMuxer;
}
}  // namespace ttv

/**
 * Interface for implementations that can handle post-encoded stream data.
 */
class ttv::broadcast::IMuxer {
 public:
  virtual ~IMuxer() = default;

  /**
   * Specifies that the encoding has begun.
   * @param parameters The parameters of the encoding.
   * @return
   *   - TTV_EC_SUCCESS if the start is successful.
   *   - Any other error code otherwise.  This will prevent the encoding stream from
   *     continuing if this is the only muxer.
   */
  virtual TTV_ErrorCode Start(const MuxerParameters& parameters) = 0;
  /**
   * Writes a video packet to the stream.
   * @param packet The video data.
   * @return
   *   - TTV_EC_SUCCESS if the writing is successful.
   *   - Any other error code otherwise.  This will stop the encoding stream if this is the only muxer.
   */
  virtual TTV_ErrorCode WriteVideoPacket(const Packet& packet) = 0;
  /**
   * Writes an audio packet to the stream.
   * @param packet The audio data.
   * @return
   *   - TTV_EC_SUCCESS if the writing is successful.
   *   - Any other error code otherwise.  This will stop the encoding stream if this is the only muxer.
   */
  virtual TTV_ErrorCode WriteAudioPacket(const Packet& packet) = 0;
  /**
   * Writes the video SPS and PPS to the stream.  This needs to be done before an IDR frame that it applies to.
   * @param sps The video SPS data.
   * @param pps The video PPS data.
   * @return
   *   - TTV_EC_SUCCESS if the writing is successful.
   *   - Any other error code otherwise.  This will stop the encoding stream if this is the only muxer.
   */
  virtual TTV_ErrorCode WriteVideoSpsPps(const std::vector<uint8_t>& sps, const std::vector<uint8_t>& pps) = 0;
  /**
   * Specifies that the encoding has stopped.
   * @return TTV_EC_SUCCESS if the stop is successful.  Any errors returned from this will be ignored.
   */
  virtual TTV_ErrorCode Stop() = 0;
};
