/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/audioconvertcontext.h"
#include "twitchsdk/broadcast/internal/audioconvert/audioconvertpipeline.h"

namespace ttv {
/**
 * An interface for converting PCM audio data from one format to another.
 *
 * Note that some of the operations during audio conversion may require some data to be maintained
 * from the previous binding. Hence, unbinding the buffer may result in some portion of the samples
 * at the end of the range being copied into an internal buffer. In order to maintain continuity of
 * the converter's output, the client must bind contiguous input buffers in chronological order.
 * Otherwise, the output data may contain gaps.
 */
class IAudioConverter {
 public:
  /**
   * Binds a buffer of input data to converter.
   * Calling this function when an input buffer is already bound to the converter is undefined
   * behavior.
   *
   * @param buffer The input buffer to bind.
   * @param range The range of values, in the input buffer format's domain, that is represented by
   *    the input buffer.
   */
  virtual void BindInputBuffer(const void* buffer, SampleRange range) = 0;

  /**
   * Returns the range, in the output buffer format's domain, that is currently available in the
   * converter. Calling this function when an input buffer is not bound is undefined behavior.
   *
   * @return The range of values that can be transferred.
   */
  virtual SampleRange GetOutputSampleRange() const = 0;

  /**
   * Transfers data to the passed output buffer.
   * Calling this function when an input buffer is not bound to the converter is undefined behavior.
   * This function will transfer data in a range that is available for reading (from the converter)
   * and writing (the passed sample range), i.e. the intersection of the two ranges. If no data was
   * transferred, this function will return a range of {0, 0}
   *
   * @param buffer The output buffer to transfer data into.
   * @param range The range of values, in the output buffer format's domain, that is represented by
   *    the output buffer.
   * @return The range of values that was written.
   */
  virtual SampleRange TransferToOutputBuffer(void* buffer, SampleRange range) = 0;

  /**
   * Unbinds the currently bound buffer of input data.
   * Calling this function when an input buffer is not bound to the converter is undefined behavior.
   */
  virtual void UnbindInputBuffer() = 0;
};

/**
 * The concrete implementation of the IAudioConverter interface.
 *
 * This class is responsible for type-erasure of configuration data for the audio converter. Options
 * and input/output formats are passed in as template parameters, but the virtual interface this
 * this class implements can be utilized without the need to keep this type information.
 *
 * @tparam InputBufferFormat A BufferFormat object specifying the PCM format of the input buffers
 *    that will be bound to this converter. See BufferFormat in dsputilities.h
 * @tparam OutputBufferFormat A BufferFormat object specifying the PCM format of the output buffers
 *    that will be written to by this converter. See BufferFormat in dsputilities.h
 * @tparam AudioConvertOptions A struct containing configuration options for audio conversion. See
 *    audioconvertoptions.h for details. If void is passed for this parameter, all default options
 *    are selected.
 */
template <typename InputBufferFormat, typename OutputBufferFormat, typename AudioConvertOptions = void>
class AudioConverter : public IAudioConverter {
 public:
  AudioConverter() : mPipeline(mContext) {}

  virtual void BindInputBuffer(const void* buffer, SampleRange range) override {
    mPipeline.BindInputBuffer(reinterpret_cast<const typename InputBufferFormat::SampleType*>(buffer), range);
  }

  virtual SampleRange GetOutputSampleRange() const override { return mPipeline.GetOutputSampleRange(); }

  virtual SampleRange TransferToOutputBuffer(void* buffer, SampleRange range) override {
    return mPipeline.TransferToOutputBuffer(reinterpret_cast<typename OutputBufferFormat::SampleType*>(buffer), range);
  }

  virtual void UnbindInputBuffer() override { mPipeline.UnbindInputBuffer(); }

 private:
  using ContextType = AudioConvertContext<AudioConvertOptions>;

  ContextType mContext;
  AudioConvertPipeline<InputBufferFormat, OutputBufferFormat, ContextType> mPipeline;
};
}  // namespace ttv
