/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/ditherers.h"
#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"
#include "twitchsdk/core/coreutilities.h"

namespace ttv {
/**
 * The following is the concept for the structure the user can provide containing the configuration
 * options for audio conversion. Any non-specified options will be filled by default values (see
 * DefaultAudioConvertOptions below).
 *
 * struct AudioConvertOptions
 * {
 *     // The ditherer to use when quantizing the bit depth of data. See ditherers.h for the available
 *     // ditherers.
 *     using Ditherer = ...;
 *
 *     // The filter cutoff to use when downsampling audio, represented as the ratio between the desired
 *     // cutoff frequency and the output sample rate. The Nyquist frequency of a digital signal (that is,
 *     // the highest frequency that can be represented at the signal's sample rate without aliasing) is
 *     // is 0.5 times the sample rate. Since the low-pass filter's frequency response function is
 *     // slightly tapered rather than a perfect rectangular function, this is usually set slightly below
 *     // 0.5.
 *     static constexpr double FilterCutoff = ...;
 *
 *     // The type of low-pass filter to use when resampling. See descriptions in FilterTypeEnum below.
 *     static constexpr FilterTypeEnum FilterType = ...;
 *
 *     // The beta factor to use when using the Kaiser-Bessel Window function.
 *     static constexpr double KaiserBeta = ...;
 *
 *     // The width of the window when using a finite impulse response low-pass filter, in input samples.
 *     static constexpr size_t FilterTapCount = ...;
 * }
 */

/**
 * The type of low pass filter to use when resampling audio.
 */
enum class FilterTypeEnum {
  /**
   * Finite impulse response filter with a Kaiser-Bessel window function.
   */
  Kaiser
};

/**
 * The default configuration options for audio conversion.
 */
struct DefaultAudioConvertOptions {
  using Ditherer = TriangularDitherer;
  static constexpr double FilterCutoff = 0.45;
  static constexpr FilterTypeEnum FilterType = FilterTypeEnum::Kaiser;
  static constexpr double KaiserBeta = 9.0;
  static constexpr size_t FilterTapCount = 16;
};

/**
 * This class, used by other internal classes, simply fills out any missing options with the defaults.
 */
template <typename PassedOptions>
struct AudioConvertOptionsDefaulter {
  template <typename Options, typename Enable = void>
  struct DithererOption {
    using Type = typename DefaultAudioConvertOptions::Ditherer;
  };

  template <typename Options>
  struct DithererOption<Options, VoidType<typename Options::Ditherer>> {
    using Type = typename Options::Ditherer;
  };

  using Ditherer = typename DithererOption<PassedOptions>::Type;

  template <typename Options, typename Enable = void>
  struct FilterCutoffOption {
    static constexpr double Value = DefaultAudioConvertOptions::FilterCutoff;
  };

  template <typename Options>
  struct FilterCutoffOption<Options, std::enable_if_t<(Options::FilterCutoff > 0.0)>> {
    static constexpr double Value = Options::FilterCutoff;
  };

  static constexpr double FilterCutoff = FilterCutoffOption<PassedOptions>::Value;

  template <typename Options, typename Enable = void>
  struct FilterTypeOption {
    static constexpr FilterTypeEnum Value = DefaultAudioConvertOptions::FilterType;
  };

  template <typename Options>
  struct FilterTypeOption<Options, std::enable_if_t<(Options::FilterType != -1)>> {
    static constexpr FilterTypeEnum Value = Options::FilterType;
  };

  static constexpr FilterTypeEnum FilterType = FilterTypeOption<PassedOptions>::Value;

  template <typename Options, typename Enable = void>
  struct KaiserBetaOption {
    static constexpr double Value = DefaultAudioConvertOptions::KaiserBeta;
  };

  template <typename Options>
  struct KaiserBetaOption<Options, std::enable_if_t<(Options::KaiserBeta > 0.0)>> {
    static constexpr double Value = Options::KaiserBeta;
  };

  static constexpr double KaiserBeta = KaiserBetaOption<PassedOptions>::Value;

  template <typename Options, typename Enable = void>
  struct TapCountOption {
    static constexpr size_t Value = DefaultAudioConvertOptions::FilterTapCount;
  };

  template <typename Options>
  struct TapCountOption<Options, std::enable_if_t<(Options::FilterTapCount > 0)>> {
    static constexpr size_t Value = Options::FilterTapCount;
  };

  static constexpr size_t FilterTapCount = TapCountOption<PassedOptions>::Value;
};
}  // namespace ttv
