/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/audioconvertoptions.h"
#include "twitchsdk/broadcast/internal/audioconvert/changefloatingpointdepthoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/converttofloatingpointoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/converttointegraltypeoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/converttosignedoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/converttounsignedoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/decreasebitdepthoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"
#include "twitchsdk/broadcast/internal/audioconvert/increasebitdepthoperator.h"
#include "twitchsdk/broadcast/internal/audioconvert/operatorchainbuilder.h"
#include "twitchsdk/broadcast/internal/audioconvert/pcmbufferaudiosource.h"
#include "twitchsdk/broadcast/internal/audioconvert/resampleoperator.h"

#include <tuple>

namespace ttv {
/**
 * A class providing a high level interface for translating between different audio formats.
 * Note that this can either support inputs and outputs having the same number of channels,
 * or a single input channel with multiple outputs (in which case the single channel is
 * duplicated across all output channels).
 *
 * @tparam InputBufferFormat The input audio format. See BufferFormat in dsputilities.h
 * @tparam OutputBufferFormat The output audio format. See BufferFormat in dsputilities.h
 * @tparam ContextType The audio context to be used with this pipeline.
 */
template <typename InputBufferFormat, typename OutputBufferFormat, typename ContextType>
class AudioConvertPipeline {
 public:
  static_assert(
    (InputBufferFormat::ChannelCount == OutputBufferFormat::ChannelCount) || (InputBufferFormat::ChannelCount == 1),
    "Multiple input channels not matching multiple output channels.");
  static constexpr size_t InputChannelCount = InputBufferFormat::ChannelCount;
  static constexpr size_t OutputChannelCount = OutputBufferFormat::ChannelCount;

  using InputSampleType = typename InputBufferFormat::SampleType;
  using OutputSampleType = typename OutputBufferFormat::SampleType;

  AudioConvertPipeline(ContextType& context) : mOperatorChains(MakeOperatorTuple<OperatorChainTuple>(context)) {}

  void BindInputBuffer(const InputSampleType* buffer, SampleRange range) { BindOperatorChains<>(buffer, range); }

  SampleRange GetOutputSampleRange() const { return std::get<0>(mOperatorChains).GetSampleRange(); }

  SampleRange TransferToOutputBuffer(OutputSampleType* buffer, SampleRange range) {
    SampleRange rangeToIterate = IntersectSampleRanges(range, std::get<0>(mOperatorChains).GetSampleRange());

    size_t bufferIndex = (rangeToIterate.startIndex - range.startIndex) * OutputChannelCount;
    for (size_t index = rangeToIterate.startIndex; index < rangeToIterate.startIndex + rangeToIterate.sampleCount;
         index++) {
      TransferSamples<>(buffer + bufferIndex, index);
      bufferIndex += OutputChannelCount;
    }
    return rangeToIterate;
  }

  void UnbindInputBuffer() { UnbindOperatorChains<>(); }

 private:
  using OperatorChainTuple = BuildOperatorChainTuple<ContextType, InputBufferFormat, OutputBufferFormat>;

  OperatorChainTuple mOperatorChains;

  template <size_t Index = 0>
    std::enable_if_t < Index<std::tuple_size<OperatorChainTuple>::value> BindOperatorChains(
                         const InputSampleType* buffer, SampleRange range) {
    using OperatorChainToBind = std::tuple_element_t<Index, OperatorChainTuple>;
    RecursiveBinder<OperatorChainToBind>::RecurseAndBind(std::get<Index>(mOperatorChains), buffer, range);

    BindOperatorChains<Index + 1>(buffer, range);
  }

  template <size_t Index = 0>
  std::enable_if_t<Index >= std::tuple_size<OperatorChainTuple>::value> BindOperatorChains(
    const InputSampleType* /*buffer*/, SampleRange /*range*/) {}

  template <typename Operator, typename Enable = void>
  struct Binder {
    static void BindOperator(Operator& /*op*/, const InputSampleType* /*buffer*/, SampleRange /*range*/) {}
  };

  template <typename Operator>
  struct Binder<Operator, VoidType<decltype(std::declval<Operator>().Bind(nullptr, {}))>> {
    static void BindOperator(Operator& op, const InputSampleType* buffer, SampleRange range) { op.Bind(buffer, range); }
  };

  template <typename Operator, typename Enable = void>
  struct RecursiveBinder {
    static void RecurseAndBind(Operator& op, const InputSampleType* buffer, SampleRange range) {
      Binder<Operator>::BindOperator(op, buffer, range);
    }
  };

  template <typename Operator>
  struct RecursiveBinder<Operator, VoidType<decltype(std::declval<Operator>().GetInputSource())>> {
    static void RecurseAndBind(Operator& op, const InputSampleType* buffer, SampleRange range) {
      Binder<Operator>::BindOperator(op, buffer, range);

      using InputSourceType = std::remove_reference_t<decltype(op.GetInputSource())>;
      RecursiveBinder<InputSourceType>::RecurseAndBind(op.GetInputSource(), buffer, range);
    }
  };

  template <size_t Index = 0>
  std::enable_if_t<(Index < std::tuple_size<OperatorChainTuple>::value) &&
                   (std::tuple_size<OperatorChainTuple>::value == 1)>
  TransferSamples(OutputSampleType* buffer, size_t inputIndex) {
    // If we have a mono input, just duplicate it across all output channels.
    OutputSampleType value = std::get<0>(mOperatorChains)[inputIndex];
    for (size_t i = 0; i < OutputChannelCount; i++) {
      buffer[i] = value;
    }
  }

  template <size_t Index = 0>
  std::enable_if_t<(Index < std::tuple_size<OperatorChainTuple>::value) &&
                   (std::tuple_size<OperatorChainTuple>::value > 1)>
  TransferSamples(OutputSampleType* buffer, size_t inputIndex) {
    buffer[Index] = std::get<Index>(mOperatorChains)[inputIndex];
    TransferSamples<Index + 1>(buffer, inputIndex);
  }

  template <size_t Index = 0>
  std::enable_if_t<Index >= std::tuple_size<OperatorChainTuple>::value> TransferSamples(
    OutputSampleType* /*buffer*/, size_t /*inputIndex*/) {}

  template <size_t Index = 0>
    std::enable_if_t < Index<std::tuple_size<OperatorChainTuple>::value> UnbindOperatorChains() {
    using OperatorChainToUnbind = std::tuple_element_t<Index, OperatorChainTuple>;
    RecursiveUnbinder<OperatorChainToUnbind>::RecurseAndUnbind(std::get<Index>(mOperatorChains));

    UnbindOperatorChains<Index + 1>();
  }

  template <size_t Index = 0>
  std::enable_if_t<Index >= std::tuple_size<OperatorChainTuple>::value> UnbindOperatorChains() {}

  template <typename Operator, typename Enable = void>
  struct Unbinder {
    static void UnbindOperator(Operator& /*op*/) {}
  };

  template <typename Operator>
  struct Unbinder<Operator, std::enable_if_t<std::is_void<decltype(std::declval<Operator>().Unbind())>::value>> {
    static void UnbindOperator(Operator& op) { op.Unbind(); }
  };

  template <typename Operator, typename Enable = void>
  struct RecursiveUnbinder {
    static void RecurseAndUnbind(Operator& op) { Unbinder<Operator>::UnbindOperator(op); }
  };

  template <typename Operator>
  struct RecursiveUnbinder<Operator, VoidType<decltype(std::declval<Operator>().GetInputSource())>> {
    static void RecurseAndUnbind(Operator& op) {
      Unbinder<Operator>::UnbindOperator(op);

      using InputSourceType = std::remove_reference_t<decltype(op.GetInputSource())>;
      RecursiveUnbinder<InputSourceType>::RecurseAndUnbind(op.GetInputSource());
    }
  };
};

template <typename InputBufferFormat, typename OutputBufferFormat, typename ContextType>
auto MakeAudioConvertPipeline(ContextType& context) {
  return AudioConvertPipeline<InputBufferFormat, OutputBufferFormat, std::decay_t<ContextType>>(context);
}
}  // namespace ttv
