/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source with one floating-point type of one depth to another.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be floating-point.
 * @tparam FloatingPointType The sample type to convert to. Must be floating-point.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossy
 *     SampleType[in] must be floating point.
 *     SampleType[out] must be floating point.
 *     SampleRate[in] == SampleRate[out]
 *     StartOffset[in] == StartOffset[out]
 *     Length[in] == Length[out]
 */
template <typename InputSource, typename FloatingPointType, typename ContextType>
class ChangeFloatingPointDepthOperator {
 public:
  ChangeFloatingPointDepthOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_floating_point<InputSampleType>::value, "Input sample type must be floating point.");

  using SampleType = FloatingPointType;
  static_assert(std::is_floating_point<SampleType>::value, "Output sample type must be floating point.");

  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    return static_cast<SampleType>(mInputSource[index]);
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
