/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source with a floating-point sample type to an integral sample type.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be floating-point.
 * @tparam IntegralType The sample type to convert to. Must be integral and signed.
 * @tparam ContextType A shared audio context type.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossy
 *     SampleType[in] must be floating point.
 *     SampleType[out] must be signed and integral.
 *     SampleRate[in] == SampleRate[out]
 *     StartOffset[in] == StartOffset[out]
 *     Length[in] == Length[out]
 */
template <typename InputSource, typename IntegralType, typename ContextType>
class ConvertToIntegralTypeOperator {
 public:
  ConvertToIntegralTypeOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_floating_point<InputSampleType>::value, "Input sample type must be floating point.");
  static_assert(std::is_integral<IntegralType>::value, "Output sample type must be integral.");
  static_assert(std::is_signed<IntegralType>::value, "Output sample type must be signed.");

  using SampleType = IntegralType;
  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    InputSampleType inputSample = mInputSource[index];

    TTV_ASSERT(inputSample <= 1.0);
    TTV_ASSERT(inputSample >= -1.0);

    InputSampleType ditheredSample = ContextType::Options::Ditherer::DitherFloatValue(
      inputSample * static_cast<InputSampleType>(BisectRange<SampleType>()));

    // Make sure to clamp the upper end of integral types, since it can't be represented by an integer.
    ditheredSample = std::min(ditheredSample, static_cast<InputSampleType>(std::numeric_limits<SampleType>::max()));
    return static_cast<SampleType>(ditheredSample);
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
