/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source with an unsigned integral sample type to a signed integral sample type of the same width.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be integral and unsigned.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossless
 *     SampleType[in] must be unsigned and integral.
 *     SampleType[out] is signed, integral and the same width as SampleType[in].
 *     SampleRate[in] == SampleRate[out]
 *     StartOffset[in] == StartOffset[out]
 *     Length[in] == Length[out]
 */
template <typename InputSource, typename ContextType>
class ConvertToSignedOperator {
 public:
  ConvertToSignedOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_unsigned<InputSampleType>::value, "Input sample type must be unsigned and integral.");

  using SampleType = std::make_signed_t<InputSampleType>;
  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    return static_cast<SampleType>(mInputSource[index] - static_cast<InputSampleType>(BisectRange<InputSampleType>()));
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
