/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source with an signed integral sample type to an unsigned integral sample type of the same width.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be integral and signed.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossless
 *     SampleType[in] must be signed and integral.
 *     SampleType[out] is unsigned, integral, and the same width as SampleType[in].
 *     SampleRate[in] == SampleRate[out]
 *     StartOffset[in] == StartOffset[out]
 *     Length[in] == Length[out]
 */
template <typename InputSource, typename ContextType>
class ConvertToUnsignedOperator {
 public:
  ConvertToUnsignedOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_signed<InputSampleType>::value, "Input sample type must be signed.");
  static_assert(std::is_integral<InputSampleType>::value, "Input sample type must be integral.");

  using SampleType = std::make_unsigned_t<InputSampleType>;
  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    return (static_cast<SampleType>(mInputSource[index]) + static_cast<SampleType>(BisectRange<SampleType>()));
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
