/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source to one with a smaller bit depth.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be integral and signed.
 * @tparam OutputSampleType The sample type to convert to. Must be integral, signed, and smaller than the input type.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossy
 *     SampleType[in] must be signed and integral.
 *     SampleType[out] must be signed and integral, and smaller than the input sample type.
 *     SampleRate[in] == SampleRate[out]
 *     StartOffset[in] == StartOffset[out]
 *     Length[in] == Length[out]
 */
template <typename InputSource, typename OutputSampleType, typename ContextType>
class DecreaseBitDepthOperator {
 public:
  DecreaseBitDepthOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_integral<InputSampleType>::value, "Input sample type must be integral.");
  static_assert(std::is_integral<OutputSampleType>::value, "Output sample type must be integral.");
  static_assert(std::is_signed<InputSampleType>::value == std::is_signed<OutputSampleType>::value,
    "Input sample type and output sample type must have same signedness.");
  static_assert(sizeof(InputSampleType) > sizeof(OutputSampleType),
    "Output sample type must be smaller than the input sample type.");

  using SampleType = OutputSampleType;
  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    // We don't directly use bit shifting math here, because bit shift right is actually
    // implementation-defined for negative values. We can use division and modulus here
    // instead and since the divisor is based on the sizeof() and known at compile-time,
    // the compiler will know that the divisor is a power of two and will do whatever
    // fancy bit shift math is appropriate for the architecture we're compiling for.
    constexpr InputSampleType divisor =
      static_cast<InputSampleType>(1 << ((sizeof(InputSampleType) - sizeof(OutputSampleType)) * 8));

    InputSampleType inputSample = mInputSource[index];

    InputSampleType quotient = inputSample / divisor;
    InputSampleType remainder = inputSample % divisor;

    // This should return us -1, 0 or 1 depending on the result of the dither.
    InputSampleType ditheredNoise = ContextType::Options::Ditherer::DitherFractionalValue(remainder, divisor);

    return static_cast<OutputSampleType>(quotient + ditheredNoise);
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
