/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/assertion.h"

#include <type_traits>

#include <cmath>

namespace ttv {
/**
 * A ditherer that simply rounds values, which is essentially no dithering at all.
 */
struct RoundingDitherer {
  template <typename FloatingPointType>
  static FloatingPointType DitherFloatValue(FloatingPointType value) {
    static_assert(std::is_floating_point<FloatingPointType>::value, "Can only dither floating point values.");

    return std::round(value);
  }

  template <typename IntegerType>
  static IntegerType DitherFractionalValue(IntegerType numerator, IntegerType denominator) {
    static_assert(std::is_integral<IntegerType>::value, "Numerator and denominator must be integral types.");
    TTV_ASSERT(denominator > 0);

    IntegerType quotient = numerator / denominator;
    IntegerType remainder = numerator % denominator;

    if (remainder >= (denominator / 2)) {
      quotient++;
    } else if (std::is_signed<IntegerType>::value && (remainder <= -(denominator / 2))) {
      quotient--;
    }

    return quotient;
  }
};

/**
 * A ditherer with a rectangular noise function.
 */
struct RectangularDitherer {
  template <typename FloatingPointType>
  static FloatingPointType DitherFloatValue(FloatingPointType value) {
    static_assert(std::is_floating_point<FloatingPointType>::value, "Can only dither floating point values.");

    return (std::rand() % 2 == 0) ? std::floor(value) : std::ceil(value);
  }

  template <typename IntegerType>
  static IntegerType DitherFractionalValue(IntegerType numerator, IntegerType denominator) {
    static_assert(std::is_integral<IntegerType>::value, "Numerator and denominator must be integral types.");
    TTV_ASSERT(denominator > 0);

    IntegerType quotient = numerator / denominator;
    IntegerType remainder = numerator % denominator;

    if (remainder > 0) {
      if (std::rand() % 2 == 0) {
        quotient++;
      }
    } else if (remainder < 0) {
      if (std::rand() % 2 == 0) {
        quotient--;
      }
    }

    return quotient;
  }
};

/**
 * A ditherer with a triangular noise function.
 */
struct TriangularDitherer {
  template <typename FloatingPointType>
  static FloatingPointType DitherFloatValue(FloatingPointType value) {
    static_assert(std::is_floating_point<FloatingPointType>::value, "Can only dither floating point values.");

    // Generate noise between -0.5 and 0.5.
    FloatingPointType noise = static_cast<double>(std::rand()) / static_cast<double>(RAND_MAX);
    noise -= 0.5;

    return std::round(value + noise);
  }

  template <typename IntegerType>
  static IntegerType DitherFractionalValue(IntegerType numerator, IntegerType denominator) {
    static_assert(std::is_integral<IntegerType>::value, "Numerator and denominator must be integral types.");
    TTV_ASSERT(denominator > 0);

    IntegerType quotient = numerator / denominator;
    IntegerType remainder = numerator % denominator;

    if (remainder > 0) {
      IntegerType noise = std::rand() % denominator;
      if (remainder + noise >= denominator) {
        quotient++;
      }
    } else if (remainder < 0) {
      IntegerType noise = std::rand() % denominator;
      if (remainder - noise < -denominator) {
        quotient--;
      }
    }

    return quotient;
  }
};
}  // namespace ttv
