/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/assertion.h"

#include <array>
#include <cmath>

namespace ttv {
constexpr double kPi = 3.14159265358979323846;

/**
 * Concept: AudioSource
 * Provides PCM samples for a range of audio on a global timeline.
 *
 * class AudioSource
 * {
 *     // The type of samples this audio source provides.
 *     using SampleType = ...;
 *
 *     // The SampleRate of this audio source, in samples per second.
 *     static constexpr size_t SampleRate = ...;
 *
 *     // The range of samples this audio source represents.
 *     SampleRange GetSampleRange() const;
 *
 *     // Retrieves the sample at the passed index from the audio source. Passing an index
 *     // outside of the audio source's sample range is undefined behavior.
 *     SampleType operator[](size_t index);
 * };
 */

/**
 *
 * Concept: Unary Audio Operator
 * Takes an InputSource and does some transformation on the represented PCM data.
 *
 * Conforms to AudioSource concept.
 *
 * template <typename InputSourceType, typename ContextType>
 * class UnaryAudioOperator
 * {
 *     UnaryAudioOperator(ContextType& context);
 *
 *     InputSource& GetInputSource();
 * }
 *
 */

/**
 * A structure that represents a range of audio samples.
 */
struct SampleRange {
  SampleRange() : startIndex(0), sampleCount(0) {}

  SampleRange(size_t startIndex, size_t sampleCount) : startIndex(startIndex), sampleCount(sampleCount) {}

  /**
   * The start of the range. That is, the number of samples preceding this range in the audio timeline.
   */
  size_t startIndex;

  /**
   * The length of the range.
   */
  size_t sampleCount;
};

/**
 * Returns the intersection of two sample ranges. If the ranges do not intersect, returns {0, 0}
 */
SampleRange IntersectSampleRanges(SampleRange firstRange, SampleRange secondRange) {
  size_t outputRangeStart = std::max(firstRange.startIndex, secondRange.startIndex);
  size_t outputRangeEnd =
    std::min(firstRange.startIndex + firstRange.sampleCount, secondRange.startIndex + secondRange.sampleCount);

  if (outputRangeStart < outputRangeEnd) {
    return {outputRangeStart, outputRangeEnd - outputRangeStart};
  } else {
    return {};
  }
}

/**
 * Performs a union of the sample ranges if and only if the ranges overlap or are contiguous.
 * If they are not, just returns the first range.
 */
SampleRange ExtendSampleRange(SampleRange range, SampleRange extension) {
  if ((extension.startIndex + extension.sampleCount >= range.startIndex) &&
      (range.startIndex + range.sampleCount >= extension.startIndex)) {
    size_t outputRangeStart = std::min(range.startIndex, extension.startIndex);
    size_t outputRangeEnd =
      std::max(range.startIndex + range.sampleCount, extension.startIndex + extension.sampleCount);

    return {outputRangeStart, outputRangeEnd - outputRangeStart};
  }

  return range;
}

/**
 * A bundle of parameters for describing the format of the data in a PCM audio buffer.
 *
 * @tparam SampleTypeArg The PCM sample type.
 * @tparam SampleRateArg The PCM sample rate, in samples per second.
 * @tparam ChannelCountArg The number of channels represented in this buffer. Currently, we only support PCM audio with
 * interleaved channels.
 */
template <typename SampleTypeArg, size_t SampleRateArg, size_t ChannelCountArg = 1>
struct BufferFormat {
  static_assert(ChannelCountArg != 0, "Channel count can't be zero.");
  static_assert(SampleRateArg != 0, "Sample rate cannot be zero.");

  using SampleType = SampleTypeArg;
  static constexpr size_t SampleRate = SampleRateArg;
  static constexpr size_t ChannelCount = ChannelCountArg;
};

/**
 * This function takes an integral type and returns a value equal to half the cardinality of that type, basically
 * 2^(number of bits of the type) / 2. This is useful in a few DSP situations. In particular, it represents the
 * centerpoint of a signal for an unsigned type. It also represents the distance from the centerpoint to the maximum
 * value of the signal for both signed and unsigned types.
 */
template <typename IntegralType>
constexpr size_t BisectRange() {
  static_assert(std::is_integral<IntegralType>::value, "Must be an integral type.");
  return 1 << ((sizeof(IntegralType) * 8) - 1);
}

/**
 * Converts a double precision floating point value to the specified type and clamps it to the expected bounds.
 * This overload is for floating point types, which should have values between -1.0 and 1.0.
 *
 * @tparam SampleType The sample type to convert to.
 */
template <typename SampleType>
std::enable_if_t<std::is_floating_point<SampleType>::value, SampleType> ClampAndCastSample(double input) {
  if (input > 1.0) {
    return static_cast<SampleType>(1.0);
  }

  if (input < -1.0) {
    return static_cast<SampleType>(-1.0);
  }

  return static_cast<SampleType>(input);
}

/**
 * Converts a double precision floating point value to the specified type and clamps it to the expected bounds.
 * This overload is for integral point types, which should have values between their maximum and minimum representable
 * values.
 *
 * @tparam SampleType The sample type to convert to.
 */
template <typename SampleType>
std::enable_if_t<std::is_integral<SampleType>::value, SampleType> ClampAndCastSample(double input) {
  if (input > std::numeric_limits<SampleType>::max()) {
    return std::numeric_limits<SampleType>::max();
  }

  if (input < std::numeric_limits<SampleType>::min()) {
    return std::numeric_limits<SampleType>::min();
  }

  return static_cast<SampleType>(input);
}

/**
 * A class that generates a table of lookup values.
 * @tparam Generator A class that conforms to the Generator concept and provides values for each entry in the table. See
 * the description of the Generator concept below.
 * @tparam Length The number of values in the table.
 */
template <typename Generator, size_t Length>
class LookupTable {
 public:
  LookupTable() {
    for (size_t i = 0; i < Length; i++) {
      mValues[i] = Generator::Generate(i, Length);
    }
  }

  typename Generator::ReturnType operator[](size_t index) const {
    TTV_ASSERT(index < mValues.size());

    return mValues[index];
  }

 private:
  std::array<typename Generator::ReturnType, Length> mValues;
};

/**
 * Generator concept:
 *
 * struct Generator
 * {
 *     using ReturnType = ...;
 *
 *     static ReturnType Generate(size_t index, size_t length);
 * };
 *
 */

/**
 * Multiplies the results of the two generators together.
 */
template <typename FirstGenerator, typename SecondGenerator>
struct ProductGenerator {
  static_assert(std::is_same<typename FirstGenerator::ReturnType, typename SecondGenerator::ReturnType>::value,
    "Factors of ProductGenerator must return the same type.");
  using ReturnType = typename FirstGenerator::ReturnType;

  static ReturnType Generate(size_t index, size_t length) {
    return FirstGenerator::Generate(index, length) * SecondGenerator::Generate(index, length);
  }
};

/**
 * Reorders the entries returned by the function to sample every Nth interval,
 * and then loop, increasing the offset each time. For example, if the sampling
 * interval is 3 over a length of 12, the new values would come from these indexes:
 *     0, 3, 6, 9, 1, 4, 7, 10, 2, 5, 8, 11
 */
template <typename InputGenerator, size_t SamplingInterval>
struct CyclingGenerator {
  using ReturnType = typename InputGenerator::ReturnType;

  static ReturnType Generate(size_t index, size_t length) {
    size_t cycleCount = length / SamplingInterval;
    size_t intervalCount = index % cycleCount;
    size_t shift = index / cycleCount;

    size_t cycledIndex = intervalCount * SamplingInterval + shift;

    return InputGenerator::Generate(cycledIndex, length);
  }
};

/**
 * Sinc function generator.
 */
template <typename SincFunctionParameters>
struct SincFunctionGenerator {
  using ReturnType = double;

  static ReturnType Generate(size_t index, size_t length) {
    if (index * 2 == length + 1) {
      // We're exactly in the center.
      return 1.0;
    }

    double time = static_cast<double>(index) / static_cast<double>(length - 1);

    // Center along the domain
    time -= 0.5;

    time *= static_cast<double>(SincFunctionParameters::Range);

    double x = 2.0 * kPi * time * SincFunctionParameters::Cutoff;

    return std::sin(x) / x;
  }
};

/**
 * Kaiser-bessel window function generator.
 */
template <typename Parameters>
struct KaiserBesselWindowFunctionGenerator {
  using ReturnType = double;

  /**
   * Zero-order modified bessel function of the first kind.
   */
  static double BesselFunction(double x) {
    constexpr double epsilon = 1E-21;
    double halfXSquared = x * x / 4.0;

    double sigmaTerm = 1.0;  // First sigma term is always 1
    double sigma = sigmaTerm;

    for (double sigmaIndex = 1.0; sigmaTerm > sigma * epsilon; sigmaIndex++) {
      // We can calculate the next term by modifying the previous term.
      sigmaTerm *= halfXSquared;
      sigmaTerm /= (sigmaIndex * sigmaIndex);

      sigma += sigmaTerm;
    }

    return sigma;
  }

  static ReturnType Generate(size_t index, size_t length) {
    double windowLengthMinusOne = static_cast<double>(Parameters::Length - 1);
    double position =
      static_cast<double>(index) * static_cast<double>(windowLengthMinusOne) / static_cast<double>(length);

    double value = 2 * position - windowLengthMinusOne;
    value /= windowLengthMinusOne;
    value *= value;
    value = 1.0 - value;
    value = sqrt(value);
    value *= Parameters::Beta;
    value = BesselFunction(value);
    value /= BesselFunction(Parameters::Beta);
    return value;
  }
};
}  // namespace ttv
