/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"

namespace ttv {
/**
 * Converts an audio source to one with a greater bit depth.
 *
 * @tparam InputSource Conforms to the AudioSource concept. Its SampleType must be integral.
 * @tparam OutputSampleType The sample type to convert to. Must be integral, larger than the input type, and have the
 * same signedness as the input type.
 *
 * Operator Type Properties:
 *     Operator Type: Unary
 *     Data Integrity: Lossless
 *     SampleType[in] must be integral.
 *     SampleType[out] must be integral, larger than the input sample type, and have the same signedness as the input
 * type. SampleRate[in] == SampleRate[out] StartOffset[in] == StartOffset[out] Length[in] == Length[out]
 */
template <typename InputSource, typename OutputSampleType, typename ContextType>
class IncreaseBitDepthOperator {
 public:
  IncreaseBitDepthOperator(ContextType& context) : mInputSource(context) {}

  using InputSampleType = typename InputSource::SampleType;
  static_assert(std::is_integral<InputSampleType>::value, "Input sample type must be integral.");
  static_assert(std::is_integral<OutputSampleType>::value, "Output sample type must be integral.");
  static_assert(std::is_signed<InputSampleType>::value == std::is_signed<OutputSampleType>::value,
    "Input sample type and output sample type must have the same signedness.");
  static_assert(sizeof(InputSampleType) < sizeof(OutputSampleType),
    "Output sample type must be larger than the input sample type.");

  using SampleType = OutputSampleType;
  static constexpr size_t SampleRate = InputSource::SampleRate;

  InputSource& GetInputSource() { return mInputSource; }

  SampleRange GetSampleRange() const { return mInputSource.GetSampleRange(); }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    size_t bitsToShift = (sizeof(OutputSampleType) - sizeof(InputSampleType)) * 8;
    return static_cast<OutputSampleType>(mInputSource[index]) << static_cast<OutputSampleType>(bitsToShift);
  }

 private:
  InputSource mInputSource;
};
}  // namespace ttv
