/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"
#include "twitchsdk/core/assertion.h"

namespace ttv {
/**
 * An AudioSource representing a channel of PCM audio backed by a buffer of (possibly interleaved) samples.
 *
 * @tparam BufferFormat A bundle of parameters that describes the format of the audio data. See BufferFormat in
 * dsputilities.h
 * @tparam SelectedChannelIndex The index of the channel containing the data. In PCM audio, the channels
 *                              are typically interleaved, so this is to ensure that we only index into
 *                              the samples for that particular channel.
 *
 */
template <typename BufferFormat, typename ContextType, size_t SelectedChannelIndex = 0>
class PCMBufferAudioSource {
 public:
  using SampleType = typename BufferFormat::SampleType;
  static constexpr size_t SampleRate = BufferFormat::SampleRate;
  static constexpr size_t ChannelCount = BufferFormat::ChannelCount;

  static_assert(SelectedChannelIndex < ChannelCount, "Selected channel index out of bounds.");

  PCMBufferAudioSource(ContextType& /*context*/) : mBuffer(nullptr) {}

  void Bind(const SampleType* buffer, SampleRange range) {
    TTV_ASSERT(buffer != nullptr);
    TTV_ASSERT(range.sampleCount != 0);

    mBuffer = buffer;
    mRange = range;
  }

  void Unbind() {
    mBuffer = nullptr;
    mRange = {};
  }

  SampleRange GetSampleRange() const { return mRange; }

  SampleType operator[](size_t index) const {
    TTV_ASSERT(index >= GetSampleRange().startIndex);
    TTV_ASSERT(index < GetSampleRange().startIndex + GetSampleRange().sampleCount);

    size_t bufferIndex = ((index - mRange.startIndex) * ChannelCount) + SelectedChannelIndex;

    return mBuffer[bufferIndex];
  }

 private:
  const SampleType* mBuffer;
  SampleRange mRange;
};
}  // namespace ttv
