/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/audioconvert/dsputilities.h"
#include "twitchsdk/core/assertion.h"

#include <limits>

namespace ttv {
/**
 * This class keeps a cache of samples in two identical ring buffers, stored back to back.
 * The goal of this is to allow efficient incremental changes to the range of cached
 * samples while always allowing reading from the cache to be entirely sequential.
 *
 * For example, imagine the double ring buffer populated like so, populated with the range
 * 5-8 and the ring buffer beginning on the 0th index (The | represents the boundary between
 * the two ring buffers):
 *
 * [5][6][7][8] | [5][6][7][8]
 *
 * But now we want to change the range to be 6-9. We can simply repopulate two entries in the
 * cache:
 *
 *  v              v
 * [9][6][7][8] | [9][6][7][8]
 *
 * Now our ring buffers start at the first index, and if we want to iterate the full range of
 * the cache (6-9), it will be sequential:
 *
 *     v  v  v     v
 * [9][6][7][8] | [9][6][7][8]
 *
 * This is more cache friendly and also allows the compiler to emit vectorized instructions
 * that are much faster than non-sequential reads.
 */
template <typename SampleType, size_t CacheSize>
class SampleCache {
 public:
  SampleCache() : mRangeStartIndex(std::numeric_limits<size_t>::max()), mRingBufferStartIndex(0) {}

  SampleRange GetRange() const { return {mRangeStartIndex, CacheSize}; }

  SampleType operator[](size_t index) {
    TTV_ASSERT(index >= mRangeStartIndex);
    TTV_ASSERT(index < mRangeStartIndex + CacheSize);

    size_t difference = index - mRangeStartIndex;
    return mDoubleRingBuffer[mRingBufferStartIndex + difference];
  }

  template <typename PopulatorType>
  void Populate(size_t startIndex, PopulatorType&& populator) {
    SampleRange rangeToPopulate;
    if (startIndex < mRangeStartIndex) {
      size_t difference = mRangeStartIndex - startIndex;
      if (difference > CacheSize) {
        // Start from scratch and repopulate the entire cache
        mRingBufferStartIndex = 0;
        mRangeStartIndex = startIndex;
        PopulateRange({startIndex, CacheSize}, std::forward<PopulatorType>(populator));
      } else {
        // Move the ring buffer index back and only populate the missing samples
        mRingBufferStartIndex += CacheSize;
        mRingBufferStartIndex -= difference;
        mRingBufferStartIndex %= CacheSize;

        mRangeStartIndex = startIndex;
        PopulateRange({startIndex, difference}, std::forward<PopulatorType>(populator));
      }
    } else if (startIndex > mRangeStartIndex) {
      size_t difference = startIndex - mRangeStartIndex;
      if (difference > CacheSize) {
        // Start from scratch and repopulate the entire cache
        mRingBufferStartIndex = 0;
        mRangeStartIndex = startIndex;
        PopulateRange({startIndex, CacheSize}, std::forward<PopulatorType>(populator));
      } else {
        // Move the ring buffer index forward and only populate the missing samples.
        mRingBufferStartIndex += difference;
        mRingBufferStartIndex %= CacheSize;
        mRangeStartIndex = startIndex;
        PopulateRange({startIndex + CacheSize - difference, difference}, std::forward<PopulatorType>(populator));
      }
    }
  }

 private:
  template <typename PopulatorType>
  void PopulateRange(SampleRange range, PopulatorType&& populator) {
    for (size_t index = range.startIndex; index < range.startIndex + range.sampleCount; index++) {
      size_t difference = index - mRangeStartIndex;
      size_t cacheIndex = (mRingBufferStartIndex + difference) % CacheSize;

      SampleType sample = static_cast<SampleType>(populator(index));
      mDoubleRingBuffer[cacheIndex] = sample;
      mDoubleRingBuffer[cacheIndex + CacheSize] = sample;
    }
  }

  std::array<SampleType, CacheSize * 2> mDoubleRingBuffer;
  size_t mRangeStartIndex;
  size_t mRingBufferStartIndex;
};
}  // namespace ttv
