/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/iaudiomixer.h"
#include "twitchsdk/core/threadsync.h"

#include <queue>

namespace ttv {
class IThread;

namespace broadcast {
class IAudioCapture;
class PcmAudioMixer;
class AudioStreamer;
}  // namespace broadcast
}  // namespace ttv

class ttv::broadcast::AudioStreamer {
 public:
  AudioStreamer();
  virtual ~AudioStreamer();

 public:
  TTV_ErrorCode SetEncoder(const std::shared_ptr<IAudioEncoder>& encoder);
  TTV_ErrorCode Initialize(const AudioParams& audioParams);
  TTV_ErrorCode StartCapture();
  void Stop();

  void SetInitialTime(uint64_t initialTime);

  float GetVolume(AudioLayerId layer) const;
  void SetVolume(AudioLayerId layer, float volume);

  void SetCapturer(AudioLayerId layerId, const std::shared_ptr<IAudioCapture>& capturer);
  void SetCapturerEnabled(AudioLayerId layer, bool enabled);
  void GetCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const;
  std::shared_ptr<IAudioCapture> GetCapturer(AudioLayerId layer) const;
  void GetEnabledCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const;
  bool HasEnabledCapturers() const;

 private:
  class CaptureContext;

  // Mapping of request time to the desired volume
  using volumeChanges_t = std::queue<std::pair<uint64_t, float>>;

  struct LayerProperties {
    LayerProperties();

    std::shared_ptr<IAudioCapture> capturer;
    volumeChanges_t volumeChanges;
    uint64_t initialTime;
    float volume;
    bool enabled;
  };

 private:
  void StartCapturers(const std::shared_ptr<IAudioMixer>& mixer);
  TTV_ErrorCode StopCapturers();
  void StopCapture();
  TTV_ErrorCode WritePcmAudioFrames(const std::shared_ptr<CaptureContext>& context, uint64_t latestTimeToProcess);
  TTV_ErrorCode WritePassThroughAudioFrames(
    const std::shared_ptr<CaptureContext>& context, uint64_t latestTimeToProcess);
  TTV_ErrorCode SubmitPcmSamplesToMixer(
    AudioLayerId audioLayerId, const int16_t* samples, size_t numSamples, uint64_t sampleTime);
  void ProcessCapturers();

 private:
  uint32_t
    mNumPcmSamplesPerChannel;  // If the audio format being captured is PCM this is the number of samples per channel.
  uint64_t mInitialTime;
  std::vector<int16_t> mMixedSamplesBuffer;

  std::shared_ptr<IThreadSync> mInitializingThreadSync;
  volatile TTV_ErrorCode mInitializeResult;

  std::map<AudioLayerId, LayerProperties> mLayers;
  std::shared_ptr<IAudioEncoder> mAudioEncoder;
  std::shared_ptr<PcmAudioMixer> mAudioMixer;
  std::shared_ptr<IThread> mDeviceProcessingThread;

  AudioParams mAudioParams;

  std::atomic_bool mDoProcessing;
};
