/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/imuxer.h"
#include "twitchsdk/broadcast/internal/muxers/flvmuxer.h"
#include "twitchsdk/broadcast/packet.h"
#include "twitchsdk/core/threadsync.h"
#include "twitchsdk/core/timer.h"

#include <atomic>
#include <mutex>
#include <queue>

namespace ttv {
class IThread;
class IMutex;

namespace broadcast {
class FrameWriter;
class FlvMuxer;
class IMuxer;
}  // namespace broadcast
}  // namespace ttv

/**
 * Responsible for gathering data packets from multiple sources and writing them to the FLV stream.
 */
class ttv::broadcast::FrameWriter : public IFrameWriter {
 public:
  enum class DelayState { Okay, Warning, Error };

  using StreamAbortFunc = std::function<void(FrameWriter* source, TTV_ErrorCode ec)>;
  using DelayStateChangedCallback = std::function<void(FrameWriter* source, DelayState state)>;
  using BandwidthStatCallback = std::function<void(FrameWriter* source, const BandwidthStat& stat)>;

 public:
  FrameWriter(bool audioEnabled);
  virtual ~FrameWriter() override;

 public:
  TTV_ErrorCode SetFlvMuxer(const std::shared_ptr<FlvMuxer>& flvMuxer);
  TTV_ErrorCode SetCustomMuxer(const std::shared_ptr<IMuxer>& muxer);

  TTV_ErrorCode Start(const VideoParams& videoParams);
  TTV_ErrorCode GetLastError();

  /**
   * Determines how many milliseconds of data is backed up on the send queue.
   */
  uint64_t GetQueueDelayInMilliseconds() const;
  /**
   * Determines how many bytes of data are backed up on the send queue.
   */
  uint32_t GetQueueDelayInBytes() const { return mNumQueuedBytes; }
  /**
   * Determines the bitrate in bits per second that should be used to maintain
   * optimal stream health.  This value will change frequently and may drop dramatically
   * if bandwidth conditions worsen.
   */
  uint32_t GetRecommendedBitRate() const;

  void SetWarningDelayThresholdMilliseconds(uint64_t warningThreshHold) { mWarningDelayThreshold = warningThreshHold; }
  void SetErrorDelayThresholdMilliseconds(uint64_t errorThreshold) { mErrorDelayThreshold = errorThreshold; }

  DelayState GetDelayState() const { return mDelayState; }
  void SetDelayStateChangedCallback(const DelayStateChangedCallback& callback) {
    mDelayStateChangedCallback = callback;
  }

  void Shutdown();

  void SetStreamAbortCallback(StreamAbortFunc func) { mStreamAbortCallback = func; }

  void SetBandwidthStatCallback(const BandwidthStatCallback& callback) { mBandwidthStatCallback = callback; }

  /**
   * Returns the average recommended kbps and the average encoded kbps since the last time this method was called.
   * @param[out] averageRecommendedKbps The average bit rate recommended by the ABS algorithm since the last time this
   * method was called.
   * @param[out] averageEncodedKbps The actual bit rate output by the video encoder since the last time this method was
   * called.
   */
  TTV_ErrorCode GatherTrackingStats(uint64_t& averageRecommendedKbps, uint64_t& averageEncodedKbps);

  // IFrameWriter implementation
  virtual TTV_ErrorCode WritePacket(std::unique_ptr<Packet>&& packet) override;

 private:
  void TransferPacketToOutputQueues(std::unique_ptr<Packet>&& packet);
  TTV_ErrorCode SendDataToMuxers();
  void UpdateRecommendedBitRate();
  void UpdateDelayState();

 private:
  std::mutex mMutex;
  std::condition_variable mCondition;
  std::queue<std::unique_ptr<Packet>> mPacketQueue;
  bool
    mWriteFrameThreadProceed;  //!< Flag to indicate that the write thread should continue processing and will be true until the writer is shutdown.

  std::shared_ptr<IThread> mWriteFrameThread;  //!< The thread that does the writing.
  std::shared_ptr<FlvMuxer> mFlvMuxer;
  std::shared_ptr<IMuxer> mCustomMuxer;
  StreamAbortFunc mStreamAbortCallback;  //!< The function to call if the stream goes down.

  std::queue<std::unique_ptr<Packet>> mAudioPacketQueue;
  std::queue<std::unique_ptr<Packet>> mVideoPacketQueue;

  VideoParams mVideoParams;

  std::atomic<uint64_t> mLastSentPacketTimestamp;
  std::atomic<uint64_t> mLastReceivedPacketTimestamp;

  DelayStateChangedCallback mDelayStateChangedCallback;
  uint64_t mWarningDelayThreshold;
  uint64_t mErrorDelayThreshold;

  std::atomic<uint32_t> mNumQueuedBytes;  //!< The number of bytes that are in the audio and video packet queues.
  std::atomic<uint64_t>
    mEncodedBitCounter;             //!< The number of bytes output by the necoder since the last adjustment time.
  uint64_t mStreamStartSystemTime;  //!< The system clock time at the start of the stream.
  uint64_t mLastStatTime;           //!< The last time a bandwidth stat was emitted.

  /**
   * This class is used to keep track of our average recommended bit rate over a period of time, to expose with the
   * `GatherTrackingStats` API. This is reset to zero each time the `GatherTrackingStats` API is called.
   */
  struct TrackedStatsRunningTotals {
    TrackedStatsRunningTotals() : recommendedBits(0), encoderOutputBits(0), elapsedSystemTime(0) {}

    uint64_t recommendedBits;
    uint64_t encoderOutputBits;
    uint64_t elapsedSystemTime;
  };
  TrackedStatsRunningTotals mRunningTotals;
  std::mutex mRunningTotalsMutex;

  BandwidthStatCallback mBandwidthStatCallback;  //!< Callback that is invoked when a new bandwidth stat is generated.

  WaitForExpiry mBitrateUpdateTimer;    //!< Timer for periodically updating the recommended bit rate.
  WaitForExpiry mBitrateIncreaseTimer;  //!< Timer that enforces the frequency at which we can increase the bit rate.
  uint32_t mRecommendedBitRate;         //!< The last recommended bitrate in bits per second.

  TTV_ErrorCode mLastError;
  DelayState mDelayState;

  bool mAudioEnabled;
};
