/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/iingesttester.h"
#include "twitchsdk/broadcast/internal/ingestsampledata.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/timer.h"

#include <functional>
#include <memory>

namespace ttv {
class User;

namespace broadcast {
class IngestTester;
class Streamer;
struct StreamerContext;
class StreamerListenerProxy;
class StreamStatsListenerProxy;
class IIngestTesterListener;
class IVideoEncoder;
class SampleDataVideoCapturer;
}  // namespace broadcast
}  // namespace ttv

/**
 * Concrete implementation of IIngestTester.
 *
 * The following properties must be set before calling Initialize().
 * - SetTestData()
 */
class ttv::broadcast::IngestTester : public UserComponent, public IIngestTester {
 public:
  using DisposerFunc = std::function<void()>;

 public:
  IngestTester(const std::shared_ptr<User>& user, const std::shared_ptr<StreamerContext>& streamerContext);
  virtual ~IngestTester() override;

  void SetDisposer(DisposerFunc&& func) { mDisposerFunc = func; }

  TTV_ErrorCode Dispose();

  void AddListener(const std::shared_ptr<IIngestTesterListener>& listener);
  void RemoveListener(const std::shared_ptr<IIngestTesterListener>& listener);

  TTV_ErrorCode SetTestData(const uint8_t* buffer, uint32_t length);

  // IIngestTester implementation
  virtual TTV_ErrorCode GetTestState(TestState& result) const override;
  virtual TTV_ErrorCode GetIngestServer(IngestServer& result) const override;
  virtual TTV_ErrorCode Start(const IngestServer& ingestServer) override;
  virtual TTV_ErrorCode Cancel() override;
  virtual TTV_ErrorCode GetMeasuredKbps(uint32_t& result) override;
  virtual TTV_ErrorCode GetTestError(TTV_ErrorCode& result) override;
  virtual TTV_ErrorCode SetTestDurationMilliseconds(uint64_t duration) override;
  virtual TTV_ErrorCode GetTestDurationMilliseconds(uint64_t& result) const override;
  virtual TTV_ErrorCode GetProgress(float& result) const override;
  virtual TTV_ErrorCode GetUserId(UserId& result) const override;

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

 protected:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  bool IsDone() const;
  void SetTestState(TestState state);
  TTV_ErrorCode StartServerTest();
  void UpdateServerTest();
  void StopServerTest();
  void UpdateProgress();
  void OnStreamerStarted();
  void OnStreamerStopped();

  EventSource<IIngestTesterListener> mListeners;
  std::shared_ptr<Streamer> mStreamer;
  std::shared_ptr<StreamerListenerProxy> mStreamerListener;
  std::shared_ptr<StreamStatsListenerProxy> mStatsListener;
  std::shared_ptr<IVideoEncoder> mVideoEncoder;
  std::shared_ptr<SampleDataVideoCapturer> mVideoCapturer;
  std::shared_ptr<StreamerContext> mStreamerContext;

  IngestServer mIngestServer;
  uint32_t mMeasuredKbps;
  TTV_ErrorCode mTestErrorCode;
  std::shared_ptr<IngestSampleData> mSampleData;

  std::shared_ptr<ComponentContainer> mComponentContainer;
  TestState mTestState;
  uint64_t mTestDurationMilliseconds;
  uint64_t mStartingBytesSent;
  uint64_t mTotalBytesSent;
  uint64_t mTotalVideoPacketsSent;
  WaitForExpiry mServerTestTimer;
  VideoParams mVideoParams;

  DisposerFunc mDisposerFunc;

  float mProgress;
  bool mBroadcasting;
  bool mWaitingForStartCallback;
  bool mWaitingForStopCallback;
};
