/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#ifndef AMF0_H
#define AMF0_H

#include <ctime>

namespace ttv {
namespace broadcast {
class IAMF0;
class AMF0Nop;
class AMF0Printer;
class AMF0StringDecoder;
class AMF0PropertyDecoder;
class AMF0NumberDecoder;

const uint8_t* DecodeAMF(const uint8_t* data, std::shared_ptr<IAMF0> output);
}  // namespace broadcast
}  // namespace ttv

/**
 * The abstract interface for writing ActionScript Message Format data.
 */
class ttv::broadcast::IAMF0 {
 public:
  enum AMFType : uint8_t {
    // Tested Types */
    number = 0x00,
    boolean = 0x01,
    string = 0x02,
    object = 0x03,
    ecmaArray = 0x08,
    objectEnd = 0x09,
    strictArray = 0x0A,
    date = 0x0B,

    // Untested but implemented types
    null = 0x05,
    undefined = 0x06,

    // Unimplemented types
    reference = 0x07,
    longString = 0x0C,
    xmlDocument = 0x0F,
    typedObject = 0x10,

    // Unsupported types
    movieclip = 0x04,  // reserved, not supported
    recordset = 0x0E,  // reserved, not supported
    unsupported = 0x0D
  };

  virtual ~IAMF0();
  virtual void Number(double value) = 0;
  virtual void Boolean(bool flag) = 0;
  virtual void String(std::string param) = 0;
  virtual void Object() = 0;
  virtual void ObjectProperty(std::string propertyName) = 0;
  virtual void Movieclip() = 0;
  virtual void Null() = 0;
  virtual void Undefined() = 0;
  virtual void Reference() = 0;
  virtual void EcmaArray(uint32_t elements) = 0;
  virtual void EcmaArrayKey(std::string keyName) = 0;
  virtual void ObjectEnd() = 0;
  virtual void StrictArray(uint32_t elements) = 0;
  virtual void Date(double date) = 0;
  virtual void LongString() = 0;
  virtual void Unsupported() = 0;
  virtual void Recordset() = 0;
  virtual void XmlDocument() = 0;
  virtual void TypedObject() = 0;
};

/**
 * The AMP writer that does nothing.
 */
class ttv::broadcast::AMF0Nop : public IAMF0 {
 public:
  virtual void Number(double /*value*/);
  virtual void Boolean(bool /*flag*/);
  virtual void String(std::string /*param*/);
  virtual void Object();
  virtual void ObjectProperty(std::string /*propertyName*/);
  virtual void Movieclip();
  virtual void Null();
  virtual void Undefined();
  virtual void Reference();
  virtual void EcmaArray(uint32_t /*elements*/);
  virtual void EcmaArrayKey(std::string /*keyName*/);
  virtual void ObjectEnd();
  virtual void StrictArray(uint32_t /*elements*/);
  virtual void Date(double /*date*/);
  virtual void LongString();
  virtual void Unsupported();
  virtual void Recordset();
  virtual void XmlDocument();
  virtual void TypedObject();
};

/**
 * The AMF writer which simply writes the data to stdout.
 */
class ttv::broadcast::AMF0Printer : public IAMF0 {
 public:
  void Number(double value) { printf("number %f\n", value); }
  void Boolean(bool flag) { printf("boolean is '%s'\n", flag ? "true" : "false"); }
  void String(std::string param) { printf("String '%s'\n", param.c_str()); }
  void Object() { printf("object START\n"); }
  void ObjectProperty(std::string propetyName) { printf("ObjectProperty '%s'\n", propetyName.c_str()); }
  void Movieclip() { printf("movieclip\n"); }
  void Null() { printf("null\n"); }
  void Undefined() { printf("undefined\n"); }
  void Reference() { printf("reference\n"); }
  void EcmaArray(uint32_t elements) { printf("ecmaArray with %u elements\n", elements); }
  void EcmaArrayKey(std::string keyName) { printf("ecmaArrayKey '%s'\n", keyName.c_str()); }
  void ObjectEnd() { printf("objectEnd\n"); }
  void StrictArray(uint32_t elements) { printf("strictArray with %u elements\n", elements); }
  void Date(double date) {
    char prettyDateTime[128];

    time_t timeSince1970 = static_cast<time_t>(date / 1000.0);

    tm* localTime = localtime(&timeSince1970);
    strftime(prettyDateTime, 128, "%c", localTime);

    printf("%s\n", prettyDateTime);
  }
  void LongString() { printf("longString\n"); }
  void Unsupported() { printf("unsupported\n"); }
  void Recordset() { printf("recordset\n"); }
  void XmlDocument() { printf("xmlDocument\n"); }
  void TypedObject() { printf("typedObject\n"); }
};

class ttv::broadcast::AMF0StringDecoder : public AMF0Nop {
 public:
  AMF0StringDecoder();

  virtual void String(std::string param);

  const std::string& GetCommandName() const { return mCommandName; }

 private:
  std::string mCommandName;
};

class ttv::broadcast::AMF0PropertyDecoder : public AMF0Nop {
 public:
  AMF0PropertyDecoder(const std::string& searchTerm);

  virtual void String(std::string param);
  virtual void ObjectProperty(std::string propertyName);

  const std::string& GetFoundValue() const { return mFoundValue; }

 private:
  AMF0PropertyDecoder operator=(const AMF0PropertyDecoder& rhs);
  const std::string mSearchTerm;
  std::string mFoundValue;
  bool mFoundProperty;
};

class ttv::broadcast::AMF0NumberDecoder : public AMF0Nop {
 public:
  AMF0NumberDecoder();

  virtual void Number(double value);

  double GetValue() const { return mValue; }

 private:
  double mValue;
};

#endif
