/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/imuxer.h"
#include "twitchsdk/broadcast/internal/muxers/flvformat.h"

#include <fstream>
#include <memory>

namespace ttv {
namespace broadcast {
// Some of this will go into a generic muxer header file
// when we have more than one muxer
struct Packet;
class FlvMuxer;
class IMuxer;
class RtmpStream;
class AMF0Encoder;
class StreamStats;
}  // namespace broadcast
}  // namespace ttv

class ttv::broadcast::FlvMuxer : public IMuxer {
 public:
  FlvMuxer(std::shared_ptr<StreamStats> streamStats);
  ~FlvMuxer() override;

  virtual void Update();
  virtual TTV_ErrorCode GetError();
  virtual TTV_ErrorCode GetAverageSendBitRate(uint64_t measurementWindowMilliseconds, uint64_t& bitsPerSecond) const;
  virtual TTV_ErrorCode GetCongestionLevel(uint64_t measurementWindowMilliseconds, double& congestionLevel) const;

  bool IsReady() { return mOutputConnected; }

  // IMuxer implementation
  virtual TTV_ErrorCode Start(const MuxerParameters& params) override;
  virtual TTV_ErrorCode WriteVideoPacket(const Packet& packet) override;
  virtual TTV_ErrorCode WriteAudioPacket(const Packet& packet) override;
  virtual TTV_ErrorCode WriteVideoSpsPps(const std::vector<uint8_t>& sps, const std::vector<uint8_t>& pps) override;
  virtual TTV_ErrorCode Stop() override;

  void SetRtmpUrl(const std::string& url) { mRtmpUrl = url; }
  void SetFlvPath(const std::wstring& path) { mFlvPath = path; }

 protected:
  TTV_ErrorCode WriteMetaPacket(const std::shared_ptr<AMF0Encoder>& encoder);
  TTV_ErrorCode WriteAudioHeader(uint8_t audioFlags, AudioFormat encodingFormat);

  TTV_ErrorCode BeginChunk(flv::TagTypes type, uint32_t timestamp, size_t length);
  TTV_ErrorCode EndChunk(size_t length);

  size_t WriteToOutput(const uint8_t* data, size_t length, bool rtmpData = true);

  template <class _T>
  size_t WriteToOutput(_T* data, bool rtmpData = true) {
    return WriteToOutput(reinterpret_cast<const uint8_t*>(data), sizeof(_T), rtmpData);
  }

  size_t WriteToOutput(const std::vector<uint8_t>& data, bool rtmpData = true) {
    return WriteToOutput(data.data(), data.size(), rtmpData);
  }

 private:
  std::string mRtmpUrl;
  std::wstring mFlvPath;
  FILE* mOutputFile;

  std::shared_ptr<StreamStats> mStreamStats;
  std::unique_ptr<RtmpStream> mRtmpStream;

  std::vector<uint8_t> mAudioFlags;
  std::vector<uint8_t> mVideoFlags;

  MuxerParameters mInitParams;
  uint64_t mTotalVideoPacketsSent;

  std::atomic_bool mOutputConnected;
};
