/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/internal/muxers/flvformat.h"
#include "twitchsdk/broadcast/internal/muxers/flvmuxer.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/thread.h"

#include <fstream>
#include <functional>
#include <future>
#include <memory>

namespace ttv {
class IThread;

namespace broadcast {
// Some of this will go into a generic muxer header file
// when we have more than one muxer
struct Packet;
class FlvMuxer;
class FlvMuxerAsync;
class RtmpStream;
class AMF0Encoder;
class StreamStats;
}  // namespace broadcast
}  // namespace ttv

class ttv::broadcast::FlvMuxerAsync : public FlvMuxer {
 public:
  FlvMuxerAsync(std::shared_ptr<StreamStats> streamStats);
  ~FlvMuxerAsync() override;

  // FlvMuxer implementation
  virtual void Update() override;
  virtual TTV_ErrorCode GetError() override;
  virtual TTV_ErrorCode GetAverageSendBitRate(
    uint64_t measurementWindowMilliseconds, uint64_t& bitsPerSecond) const override;
  virtual TTV_ErrorCode GetCongestionLevel(
    uint64_t measurementWindowMilliseconds, double& congestionLevel) const override;

  // IMuxer implementation
  virtual TTV_ErrorCode Start(const MuxerParameters& params) override;
  virtual TTV_ErrorCode WriteVideoPacket(const Packet& packet) override;
  virtual TTV_ErrorCode WriteAudioPacket(const Packet& packet) override;
  virtual TTV_ErrorCode Stop() override;

 private:
  using ExecuteFunction = std::function<TTV_ErrorCode()>;
  class Task {
   public:
    Task(ExecuteFunction executeFunctionPromise);

    TTV_ErrorCode GetResultFromFuture();
    void Run();

   private:
    ExecuteFunction mExecuteFunction;
    std::promise<TTV_ErrorCode> mResultPromise;
  };

  void StartProcessThread();
  void StopProcessThread();
  void ProcessQueue();
  void ExecuteTask(std::shared_ptr<Task> task) const;
  void ExecuteAsync(ExecuteFunction func) const;
  TTV_ErrorCode ExecuteSyncWithResult(ExecuteFunction func) const;

  std::atomic_bool mDoProcessing;
  mutable std::condition_variable mProcessQueueCond;
  mutable std::mutex mProcessQueueMutex;
  mutable std::queue<std::shared_ptr<Task>> mProcessQueue;
  std::shared_ptr<IThread> mProcessThread;
};
