#pragma once

#include "twitchsdk/broadcast/internal/muxers/flvformat.h"

namespace ttv {
namespace broadcast {
enum { RTMP_DEFAULT_CHUNK_SIZE = 128 };

/// Chunk header-type, indicated in first byte of chunk basic header
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum Chunktype : uint8_t { RTMP_CHUNKTYPE_LARGE = 0, RTMP_CHUNKTYPE_MEDIUM, RTMP_CHUNKTYPE_SMALL, RTMP_CHUNKTYPE_NONE };

/// Common Chunk-stream-ids. Not set in stone- they are just the usual convention...
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum Channel : uint8_t {
  RTMP_CHANNEL_INVALID = 0,
  RTMP_CHANNEL_NETWORK = 2,
  RTMP_CHANNEL_SYSTEM,
  RTMP_CHANNEL_SOURCE,
  RTMP_CHANNEL_NETSTREAM,
  RTMP_CHANNEL_VIDEO = 8,
  RTMP_CHANNEL_AUDIO
};

/// Packet types, indicated in the chunk header
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum PacketType : uint8_t {
  RTMP_PKT_INVALID = 0x00,  ///< Not a real value according to the standard
  RTMP_PKT_CHUNK_SIZE = 0x01,
  RTMP_PKT_ABORT_MSG = 0x02,
  RTMP_PKT_BYTES_READ = 0x03,
  RTMP_PKT_CONTROL = 0x04,  ///< See RTMP_CTL_* below for possible values
  RTMP_PKT_WINACKSIZE = 0x05,
  RTMP_PKT_PEER_BW = 0x06,

  RTMP_PKT_EDGE_ORIGIN = 0x07,  ///< No payloads actually documented...

  RTMP_PKT_AUDIO = 0x08,
  RTMP_PKT_VIDEO = 0x09,

  RTMP_PKT_AMF3_DATA = 0x0F,  ///< amf3 metadata body
  RTMP_PKT_AMF3_SO = 0x10,    ///< amf3 shared object command (don't support)
  RTMP_PKT_AMF3 = 0x11,       ///< cmd-name, trans-id, amf3 body

  RTMP_PKT_AMF0_DATA = 0x12,  ///< amf0 metadata body
  RTMP_PKT_AMF0_SO = 0x13,    ///< amf0 shared object command (don't support)
  RTMP_PKT_AMF0 = 0x14,       ///< cmd-name, trans-id, amf0 body

  RTMP_PKT_AGGREGATE = 0x16  ///< needs to be chopped up
};

/// When packet type is RTMP_PKT_CONTROL- these are the possible payloads
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum CtlType : uint8_t {
  RTMP_CTL_STREAM_BEGIN = 0,   ///< Data: stream-id (4B)
  RTMP_CTL_STREAM_EOF,         ///< Data: stream-id (4B)
  RTMP_CTL_STREAM_DRY,         ///< Data: stream-id (4B)
  RTMP_CTL_SET_BUFFER_LENGTH,  ///< Data: stream-id, buffer-length in milliseconds (*** 8B ***)
  RTMP_CTL_STREAM_RECORDED,    ///< Server telling client it's not live, Data: stream-id (4B)
  RTMP_CTL_PING = 6,           ///< Data: timestamp (4B)
  RTMP_CTL_PONG                ///< Responds with the ping's timestamp, so RTT can be calculated (4B)
};

//////////////////////////////////////////////////////////////////////////
template <class Inserter>
inline void InsertBigEndian32(Inserter it, uint32_t val) {
  auto p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[3];
  *it++ = p[2];
  *it++ = p[1];
  *it++ = p[0];
}

template <class Inserter>
inline void InsertBigEndian(Inserter it, uint32_t val) {
  auto p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[2];
  *it++ = p[1];
  *it++ = p[0];
}

template <class Inserter>
inline void InsertBigEndian(Inserter it, uint16_t val) {
  auto p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[1];
  *it++ = p[0];
}

inline uint32_t BigToLittle(uint32_t source) {
  uint32_t target = 0;

  auto pSource = reinterpret_cast<uint8_t*>(&source);
  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = pSource[3];
  pTarget[1] = pSource[2];
  pTarget[2] = pSource[1];
  pTarget[3] = pSource[0];

  return target;
}

inline uint32_t BigToLittle(const flv::uint24_t source) {
  uint32_t target = 0;

  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = source[2];
  pTarget[1] = source[1];
  pTarget[2] = source[0];

  return target;
}
//////////////////////////////////////////////////////////////////////////

struct ChunkHeader {
  Chunktype type;
  uint8_t streamId;
  uint32_t timestamp;
  uint32_t packetLength;
  PacketType messageType;
  uint32_t messageStreamId;
};

inline size_t GetChunkSize(char firstByteOfChunk) {
  Chunktype type = static_cast<Chunktype>(firstByteOfChunk >> 6);

  // if this assert hits we got a really big chunk header
  assert((firstByteOfChunk & 0x3f) != 0x3f);

  switch (type) {
    case RTMP_CHUNKTYPE_SMALL:
      return 4;
    case RTMP_CHUNKTYPE_MEDIUM:
      return 8;
    case RTMP_CHUNKTYPE_LARGE:
      return 12;
    case RTMP_CHUNKTYPE_NONE:
    default:
      return 1;
  }
}

inline ChunkHeader PopulateChunkHeader(const uint8_t* buffer) {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wold-style-cast"
  int pos = 0;

  ChunkHeader h;
  h.type = static_cast<Chunktype>(buffer[pos] >> 6);
  h.streamId = buffer[pos++] & 0x3f;

  if (h.type != RTMP_CHUNKTYPE_NONE) {
    const flv::uint24_t* as24 = reinterpret_cast<const flv::uint24_t*>(&buffer[pos]);
    pos += 3;
    h.timestamp = BigToLittle(*as24);
  }
  if (h.type < RTMP_CHUNKTYPE_SMALL) {
    const flv::uint24_t* as24 = reinterpret_cast<const flv::uint24_t*>(&buffer[pos]);
    pos += 3;
    h.packetLength = BigToLittle(*as24);
    h.messageType = static_cast<PacketType>(buffer[pos++]);
  }
  if (h.type == RTMP_CHUNKTYPE_LARGE) {
    uint8_t* p = reinterpret_cast<uint8_t*>(&h.messageStreamId);
    memcpy(p, buffer + pos, 4);
  }

  return h;
#pragma clang diagnostic pop
}

using RtmpMessageheader = uint8_t[16];

class RtmpMessageDetails {
 public:
  enum State { Invalid, New, Used };

  RtmpMessageDetails() : mState(Invalid) {}

  ~RtmpMessageDetails() { TTV_ASSERT(mState != Used || mDataWritten == mPacketLength); }

  RtmpMessageDetails(
    Channel chunkChannel, uint32_t timestamp, uint32_t packetLength, PacketType messageType, uint32_t messageStreamId)
      : mChunkChannel(chunkChannel),
        mTimestamp(timestamp),
        mPacketLength(packetLength),
        mMessageType(messageType),
        mMessageStreamId(messageStreamId),
        mState(New),
        mDataWritten(0) {}

  RtmpMessageDetails(const RtmpMessageDetails& rhs) { *this = rhs; }

  RtmpMessageDetails& operator=(const RtmpMessageDetails& rhs) {
    if (&rhs != this) {
      memcpy(this, &rhs, sizeof(RtmpMessageDetails));
    }
    return *this;
  }

  inline bool IsNew() const { return mState == New; }
  inline bool IsValid() const { return mState != Invalid; }
  uint32_t Length() const { return mPacketLength; }
  void DataWritten(uint32_t length) {
    TTV_ASSERT(mState == Used);
    mDataWritten += length;
  }

  inline size_t PackMessageHeader(RtmpMessageheader header) {
    assert(mState == Used || mState == New);
    if (mState == Invalid) {
      return 0;
    }

    auto type = mState == New ? RTMP_CHUNKTYPE_LARGE : RTMP_CHUNKTYPE_NONE;
    assert(mChunkChannel != RTMP_CHANNEL_INVALID);
    size_t pos = 0;

    header[pos++] = static_cast<uint8_t>((type << 6) + mChunkChannel);

    uint32_t shortTimestamp = std::min(0xFFFFFFU, mTimestamp);

    if (type != RTMP_CHUNKTYPE_NONE) {
      auto p = reinterpret_cast<uint8_t*>(&shortTimestamp);
      header[pos++] = p[2];
      header[pos++] = p[1];
      header[pos++] = p[0];
    }
    if (type < RTMP_CHUNKTYPE_SMALL) {
      auto p = reinterpret_cast<uint8_t*>(&mPacketLength);
      header[pos++] = p[2];
      header[pos++] = p[1];
      header[pos++] = p[0];
      header[pos++] = mMessageType;
    }

    if (type == RTMP_CHUNKTYPE_LARGE) {
      auto p = reinterpret_cast<uint32_t*>(&header[pos]);
      *p = mMessageStreamId;
      pos += 4;
    }

    if (shortTimestamp == 0xFFFFFF) {
      auto p = reinterpret_cast<uint8_t*>(&mTimestamp);
      header[pos++] = p[3];
      header[pos++] = p[2];
      header[pos++] = p[1];
      header[pos++] = p[0];
    }

    mState = Used;
    return pos;
  }

 private:
  Channel mChunkChannel;
  uint32_t mTimestamp;
  uint32_t mPacketLength;
  PacketType mMessageType;
  uint32_t mMessageStreamId;
  State mState;
  uint32_t mDataWritten;
};

/*
        inline size_t CreateMessageHeader (RtmpMessageheader header,
            Chunktype type,
            ttv::Channel chunkChannel,
            uint32_t timestamp,
            uint32_t packetLength,
            PacketType messageType,
            uint32_t messageStreamId)
        {
            assert (chunkChannel != RTMP_CHANNEL_INVALID);
            size_t pos = 0;

            header[pos++] = (type << 6) + chunkChannel;

            if (type != RTMP_CHUNKTYPE_NONE)
            {
                auto p = reinterpret_cast<uint8_t*>(&timestamp);
                header[pos++] = p[2];
                header[pos++] = p[1];
                header[pos++] = p[0];
            }
            if (type < RTMP_CHUNKTYPE_SMALL)
            {
                auto p = reinterpret_cast<uint8_t*>(&packetLength);
                header[pos++] = p[2];
                header[pos++] = p[1];
                header[pos++] = p[0];
                header[pos++] = messageType;
            }

            if (type == RTMP_CHUNKTYPE_LARGE)
            {
                auto p = reinterpret_cast<uint32_t*>(&header[pos]);
                *p = messageStreamId;
                pos+=4;
            }

            return pos;
        }
        */
}  // namespace broadcast
}  // namespace ttv
