/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/internal/muxers/amf0encoder.h"
#include "twitchsdk/core/socket.h"

namespace ttv {
namespace broadcast {
struct RtmpContext;
class StreamStats;
}  // namespace broadcast
}  // namespace ttv

//////////////////////////////////////////////////////////////////////////
// Context for RTMP operations.
//
// Making this a struct and not a class because it's usage is encapsulated
// by designed. Nothing has access to this except for the states themselves
//////////////////////////////////////////////////////////////////////////

struct ttv::broadcast::RtmpContext {
 public:
  enum class State {
    Invalid = -1,
    Idle,
    Initialize,
    Handshake,
    Connect,
    CreateStream,
    Publish,
    SendVideo,
    Shutdown,
    Error,

    Count
  };

 public:
  RtmpContext(std::shared_ptr<StreamStats> streamStats);

  std::string mURL;
  std::string mHostName;
  std::string mApplication;
  uint32_t mPort;
  std::string mStreamName;

  static const uint32_t kDesiredOutgoingChunkSize = 4096;
  uint mIncomingChunkSize;
  uint mOutgoingChunkSize;
  uint mWindowAckSize;  // TODO: Should we also ack?
  uint mIncomingPeerBW;

  size_t mChunkSpace;  // How much more space do we have in the current chunk until a new header is needed

  double mTransactionId;
  double mStreamId;

  TTV_ErrorCode mLastError;
  BufferedSocket mSocket;

  AMF0Encoder mAmfEncoder;

  std::shared_ptr<StreamStats> mStreamStats;

  void SetNextState(State nextState);

  void SetCurrentStateToNext() {
    OnStateChange();
    mCurrentState = mNextState;
    ttv::trace::Message("rtmp", MessageLevel::Info, "Setting current state to %d", mCurrentState);
  }

  State GetCurrentState() const { return mCurrentState; }
  State GetNextState() const { return mNextState; }

 private:
  State mCurrentState;
  State mNextState;

  void OnStateChange();
};
