/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/internal/framewriter.h"
#include "twitchsdk/broadcast/internal/task/channelinfotask.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"

namespace ttv {
class TaskRunner;
class TrackingContext;

namespace broadcast {
class Streamer;
struct StreamerContext;
class AudioStreamer;
class FlvMuxer;
class FlvMuxerAsync;
class FrameWriter;
class StreamStats;
class VideoFrame;
class VideoStreamer;
class IVideoEncoder;
class IVideoCapture;
class IAudioCapture;
class IAudioMixer;
class StreamerListenerProxy;
class IBandwidthStatListener;
class IMuxer;
}  // namespace broadcast
}  // namespace ttv

struct ttv::broadcast::StreamerContext {
  StreamerContext() : isBusy(false), connectionType(ConnectionType::Unknown) {}

  std::atomic_bool isBusy;
  std::shared_ptr<TrackingContext> sharedTrackingContext;
  ConnectionType connectionType;
  std::string category;
  std::string title;
};

/**
 * The following must be set before initialization:
 *
 * - SetEncoders()
 * - SetChannelName()
 *
 * The starting and stopping of the broadcast is always asynchronous.  When stopping a broadcast it is possible that
 * the stop may not complete immediately of the network has backed up and encoded frames are still being sent to the
 * ingest server.  Wait for the stop callback.
 */
class ttv::broadcast::Streamer : public ttv::UserComponent {
 public:
  enum class StreamerState { Stopped, Starting, Started, Stopping };

  struct StartParams {
    VideoParams videoParams;
    IngestServer ingestServer;
    StreamStartFlags flags;
    std::wstring outputFile;
    bool enableAsyncFlvMuxer;
  };

  using StartCallback = std::function<void(TTV_ErrorCode ec)>;
  using StopCallback = std::function<void(TTV_ErrorCode ec)>;
  using FrameWriterDelayStateChangedCallback = std::function<void(FrameWriter::DelayState state)>;

  class IListener {
   public:
    virtual ~IListener() = default;

    /**
     * Fired after a call to Start() has been serviced.
     */
    virtual void OnStartComplete(Streamer* source, TTV_ErrorCode ec) = 0;
    /**
     * Fired after a call to Stop() has been serviced.
     */
    virtual void OnStopComplete(Streamer* source, TTV_ErrorCode ec) = 0;
    /**
     * Fired any time the stream state changes.
     */
    virtual void OnStreamerStateChanged(Streamer* source, StreamerState state, TTV_ErrorCode ec) = 0;
    /**
     * Fired periodically when there are issues with the user's bandwidth while broadcasting.
     */
    virtual void OnBandwidthWarning(Streamer* source, TTV_ErrorCode ec, uint32_t backupMilliseconds) = 0;
    /**
     * Fired when we fetch the broadcast_id after the broadcast has started.
     */
    virtual void OnStreamInfoFetched(Streamer* source, TTV_ErrorCode ec, const StreamInfo& streamInfo) = 0;
    /**
     * Fired when a StreamKey request returns a CanThey Error.
     */
    virtual void OnStreamKeyError(Streamer* source, const CanTheyError& error) = 0;
  };

 public:
  Streamer(const std::shared_ptr<User>& user, const std::shared_ptr<StreamerContext>& context);
  virtual ~Streamer() override;

  void AddListener(const std::shared_ptr<IListener>& listener);
  void RemoveListener(const std::shared_ptr<IListener>& listener);

  void AddBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener);
  void RemoveBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener);

  /**
   * These must be set before calling Initialize().
   */
  TTV_ErrorCode SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder);
  TTV_ErrorCode SetAudioEncoder(const std::shared_ptr<IAudioEncoder>& encoder);
  TTV_ErrorCode SetCustomMuxer(const std::shared_ptr<IMuxer>& muxer);

  /**
   * Sets the broadcaster_software value for the stream.
   */
  void SetBroadcasterSoftware(const std::string& str) { mBroadcasterSoftware = str; }

  /**
   * Sets the force archive flag for the stream
   */
  void SetForceArchiveBroadcast(bool forceArchive) { mForceArchiveBroadcast = forceArchive; }

  /**
   * Sets the don't force archive flag for the stream
   */
  void SetForceDontArchiveBroadcast(bool forceDontArchive) { mForceDontArchiveBroadcast = forceDontArchive; }

  void SetFrameWriterDelayStateChangedCallback(const FrameWriterDelayStateChangedCallback& callback) {
    mFrameWriterDelayStateChangedCallback = callback;
  }

  // Component overrides
  virtual void SetTaskRunner(std::shared_ptr<TaskRunner> taskRunner) override;
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  // UserComponent overrides
  virtual std::string GetLoggerName() const override;
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  /**
   * Performs an asynchronous start.
   */
  TTV_ErrorCode Start(const StartParams& params, StartCallback&& callback);

  /**
   * Performs an asynchronous stop.
   */
  TTV_ErrorCode Stop(const std::string& reason, StopCallback&& callback);

  TTV_ErrorCode GetVolume(AudioLayerId layer, float& volume) const;
  TTV_ErrorCode SetVolume(AudioLayerId layer, float volume);
  void SetAudioCapturer(AudioLayerId layerId, const std::shared_ptr<IAudioCapture>& capturer);
  void SetAudioCapturerEnabled(AudioLayerId layer, bool enabled);
  void GetAudioCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const;
  std::shared_ptr<IAudioCapture> GetAudioCapturer(AudioLayerId layer) const;

  void SetVideoCapturer(std::shared_ptr<IVideoCapture> capturer);
  std::shared_ptr<IVideoCapture> GetVideoCapturer() const;

  uint64_t GetStreamTime() const;

  std::shared_ptr<StreamStats> GetStreamStats() { return mStreamStats; }
  std::shared_ptr<const StreamStats> GetStreamStats() const { return mStreamStats; }

  VideoParams GetVideoParams() const { return mStartParams.videoParams; }

  TTV_ErrorCode ValidateParams(const VideoParams& videoParams, const IngestServer& ingestServer) const;

  StreamerState GetStreamerState() const;

  // TTV_ErrorCode SendMetaData(const char* authToken,
  //                         TwitchAPI::MetaDataEventClass eventClass, const char* name,
  //                         int64_t streamTime, unsigned long sequenceNumber,
  //                         const char* humanDescription, const char* data,
  //                         TTV_TaskCallback callback, void* userData);

  static std::string GetComponentName() { return "ttv::Streamer"; }
  static void GetRequiredAuthScopes(std::vector<std::string>& scopes);

 private:
  TTV_ErrorCode GetStreamKey(std::function<void(TTV_ErrorCode ec)>&& callback);
  TTV_ErrorCode AddVideoFrame(const std::shared_ptr<VideoFrame>& frame);
  TTV_ErrorCode GetStreamInfo();
  TTV_ErrorCode SetIngestServer(const IngestServer& ingestServer);
  TTV_ErrorCode SetStreamName(const std::string& streamKey, bool absEnabled);

  TTV_ErrorCode KickOffStart(const StartParams& params, StartCallback&& callback);
  TTV_ErrorCode InternalStart(const StartParams& params);
  TTV_ErrorCode InternalStop(TTV_ErrorCode ec, bool solicited, StopCallback&& callback);
  TTV_ErrorCode UpdateBandwidthWarningState();

  /**
   * Sends a tracking event indicating that we failed to start the broadcast.
   * @param[in] ec The error that occurred when trying to start.
   * @param[in] synchronous If this is true, the failure occurred synchronously and an error was immediately returned to
   * the client. If this is false, the error was delivered asynchronously in the client's callback.
   */
  TTV_ErrorCode TrackStartFailure(TTV_ErrorCode ec, bool synchronous);

  /**
   * Sends a tracking event indicating that the broadcast failed mid-stream.
   * @param[in] ec The error that caused the stream failure.
   */
  TTV_ErrorCode TrackStreamFailure(TTV_ErrorCode ec);
  TTV_ErrorCode TrackMinuteBroadcast();

  void NotifyStreamerStateChanged();

 private:
  std::unique_ptr<StreamInfo> mStreamInfo;

 private:
  StartParams mStartParams;
  AudioParams mAudioParams;
  IngestServer mSelectedIngestServer;
  std::shared_ptr<StreamerContext> mContext;
  std::string mOutputStreamName;
  std::shared_ptr<FlvMuxer> mFlvMuxer;
  std::shared_ptr<FrameWriter> mFrameWriter;
  std::shared_ptr<StreamStats> mStreamStats;
  std::shared_ptr<VideoStreamer> mVideoStreamer;
  std::shared_ptr<AudioStreamer> mAudioStreamer;
  EventSource<IListener> mListeners;
  ttv::ConcurrentQueue<BandwidthStat> mStatQueue;
  EventSource<IBandwidthStatListener> mBandwidthStatListeners;

  ChannelId mChannelId;    //!< The name of the channel to broadcast to.
  std::string mStreamKey;  //!< The key that grants stream access to the channel.

  std::string mBroadcasterSoftware;  //!< The broadcaster parameter to send over RTMP.
  uint64_t mInitialTime;             //!< The number of system ticks at the time of stream start.
  mutable std::mutex mStateMutex;
  StreamerState mStreamerState;
  RetryTimer mStreamInfoFetchTimer;
  StreamerState mLastReportedStreamerState;  //!< The state last reported to the client
  TTV_ErrorCode mStateChangeError;           //!< The cause of the last state change.
  WaitForExpiry mBandwidthWarningTimer;
  TTV_ErrorCode mBandwidthWarningState;

  WaitForExpiry mMinuteBroadcastTrackingTimer;

  std::shared_ptr<IVideoEncoder> mVideoEncoder;
  std::shared_ptr<IAudioEncoder> mAudioEncoder;

  std::shared_ptr<IVideoCapture> mVideoCapturer;

  std::shared_ptr<IMuxer> mCustomMuxer;
  std::shared_ptr<TrackingContext> mTrackingContext;

  FrameWriterDelayStateChangedCallback mFrameWriterDelayStateChangedCallback;

  bool mFirstFrameSubmitted;
  bool mBandwidthTestMode;
  bool
    mForceArchiveBroadcast;  //!< If true appends recorder=1 to the RTMP URL forcing the server to record the broadcast
  bool
    mForceDontArchiveBroadcast;  //!< If true appends recorder=0 to the RTMP URL forcing the server to not record the broadcast
};

/**
 * A lambda proxy for Streamer::IListener.
 */
class ttv::broadcast::StreamerListenerProxy : public Streamer::IListener {
 public:
  using StartCompleteFunc = std::function<void(Streamer* source, TTV_ErrorCode ec)>;
  using StopCompleteFunc = std::function<void(Streamer* source, TTV_ErrorCode ec)>;
  using StreamerStateChangedFunc =
    std::function<void(Streamer* source, Streamer::StreamerState state, TTV_ErrorCode ec)>;
  using BandwidthWarningFunc = std::function<void(Streamer* source, TTV_ErrorCode ec, uint32_t backupMilliseconds)>;
  using StreamInfoFetchedFunc = std::function<void(Streamer* source, TTV_ErrorCode ec, const StreamInfo& streamInfo)>;
  using StreamKeyErrorFunc = std::function<void(Streamer* source, const CanTheyError& error)>;

 public:
  virtual void OnStartComplete(Streamer* source, TTV_ErrorCode ec) override {
    if (mStartCompleteFunc != nullptr) {
      mStartCompleteFunc(source, ec);
    }
  }

  virtual void OnStopComplete(Streamer* source, TTV_ErrorCode ec) override {
    if (mStopCompleteFunc != nullptr) {
      mStopCompleteFunc(source, ec);
    }
  }

  virtual void OnStreamerStateChanged(Streamer* source, Streamer::StreamerState state, TTV_ErrorCode ec) override {
    if (mStreamerStateChangedFunc != nullptr) {
      mStreamerStateChangedFunc(source, state, ec);
    }
  }

  virtual void OnBandwidthWarning(Streamer* source, TTV_ErrorCode ec, uint32_t backupMilliseconds) override {
    if (mBandwidthWarningFunc != nullptr) {
      mBandwidthWarningFunc(source, ec, backupMilliseconds);
    }
  }

  virtual void OnStreamInfoFetched(Streamer* source, TTV_ErrorCode ec, const StreamInfo& streamInfo) override {
    if (mStreamInfoFetchedFunc != nullptr) {
      mStreamInfoFetchedFunc(source, ec, streamInfo);
    }
  }

  virtual void OnStreamKeyError(Streamer* source, const CanTheyError& error) override {
    if (mStreamInfoFetchedFunc != nullptr) {
      mStreamKeyErrorFunc(source, error);
    }
  }

 public:
  StartCompleteFunc mStartCompleteFunc;
  StopCompleteFunc mStopCompleteFunc;
  StreamerStateChangedFunc mStreamerStateChangedFunc;
  BandwidthWarningFunc mBandwidthWarningFunc;
  StreamInfoFetchedFunc mStreamInfoFetchedFunc;
  StreamKeyErrorFunc mStreamKeyErrorFunc;
};
