/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/eventsource.h"

namespace ttv {
namespace broadcast {
class StreamStats;
class StreamStatsListenerProxy;
}  // namespace broadcast
}  // namespace ttv

class ttv::broadcast::StreamStats {
 public:
  enum class StatType {
    RtmpState,             //!< The value should be interpretted as RtmpState.
    RtmpTotalBytesSent,    //!< The total number of bytes that have been sent over the socket via RTMP.
    TotalVideoPacketsSent  //!< The total number of video packets sent.
  };

  class IListener {
   public:
    virtual ~IListener() = default;
    virtual void OnStatReceived(StreamStats* source, StatType type, uint64_t data) = 0;
  };

 public:
  /**
   * Must be called from the same thread as Flush().
   */
  void AddListener(const std::shared_ptr<IListener>& listener);
  /**
   * Must be called from the same thread as Flush().
   */
  void RemoveListener(const std::shared_ptr<IListener>& listener);
  /**
   * Flushes all stats in the registered listeners.
   */
  void Flush();
  /**
   * Reports a new metric instance.  This can be called from any thread.
   */
  void Add(StatType type, uint64_t data);

 private:
  using value_t = std::pair<StatType, uint64_t>;
  ConcurrentQueue<value_t> mQueue;
  EventSource<IListener> mListeners;
};

/**
 * A lambda proxy for StreamStats::IListener.
 */
class ttv::broadcast::StreamStatsListenerProxy : public StreamStats::IListener {
 public:
  using OnStatReceivedFunc = std::function<void(StreamStats* source, StreamStats::StatType type, uint64_t data)>;

 public:
  // StreamStats::Listener implementation
  virtual void OnStatReceived(StreamStats* source, StreamStats::StatType type, uint64_t data) override {
    if (mOnStatReceivedFunc != nullptr) {
      mOnStatReceivedFunc(source, type, data);
    }
  }

 public:
  OnStatReceivedFunc mOnStatReceivedFunc;
};
