/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/core/component.h"

#include <string>

namespace ttv {
namespace broadcast {
class TwitchAPI;
class MatchGameNamesTask;
}  // namespace broadcast
}  // namespace ttv

/**
 * A convenience component for managing requests.  Some requests are not associated with a User so this is not a proper
 * UserComponent since it needs to be hosted independently from a User.
 */
class ttv::broadcast::TwitchAPI : public ttv::Component {
 public:
  TwitchAPI();
  TwitchAPI(const std::shared_ptr<User>& user);

  static std::string GetComponentName() { return "ttv::TwitchAPI"; }

  virtual std::string GetLoggerName() const override;

 public:
  struct ChannelInfo {
    std::string streamKey;
    std::string loginName;
    std::string displayName;
    std::string url;
  };

  enum { kDefaultRequestTimeoutInSec = 10 };

  enum MetaDataEventClass {
    METADATA_ACTION,
    METADATA_START,
    METADATA_END,
  };

 public:
  static const char* CurrentApiVersionString();

 public:
  using GetChannelInfoCallback = std::function<void(TwitchAPI* source, TTV_ErrorCode ec, ChannelInfo&& result)>;
  TTV_ErrorCode GetChannelInfo(GetChannelInfoCallback&& callback);

  using SetStreamInfoCallback = std::function<void(TwitchAPI* source, TTV_ErrorCode ec)>;
  TTV_ErrorCode SetStreamInfo(
    ChannelId channelId, const std::string& streamTitle, const std::string& gameName, SetStreamInfoCallback&& callback);

  using GetStreamInfoCallback =
    std::function<void(TwitchAPI* source, TTV_ErrorCode ec, const std::shared_ptr<StreamInfo>& result)>;
  TTV_ErrorCode GetStreamInfo(ChannelId channelId, GetStreamInfoCallback&& callback);

  using GetChannelRecordingStatusCallback =
    std::function<void(TwitchAPI* source, TTV_ErrorCode ec, const std::shared_ptr<ArchivingState>& result)>;
  TTV_ErrorCode GetChannelRecordingStatus(GetChannelRecordingStatusCallback&& callback);

  using RunCommercialCallback = std::function<void(TwitchAPI* source, TTV_ErrorCode ec)>;
  TTV_ErrorCode RunCommercial(ChannelId channelId, uint32_t timeBreakSeconds, RunCommercialCallback&& callback);

  using GetGameNameListCallback = std::function<void(
    TwitchAPI* source, const std::string& str, TTV_ErrorCode ec, const std::shared_ptr<GameInfoList>& result)>;
  TTV_ErrorCode GetGameNameList(const std::string& str, GetGameNameListCallback&& callback);

  using GetIngestServerListCallback =
    std::function<void(TwitchAPI* source, TTV_ErrorCode ec, std::vector<IngestServer>&& result)>;
  TTV_ErrorCode GetIngestServerList(GetIngestServerListCallback&& callback);

 private:
  std::weak_ptr<User> mUser;
  std::shared_ptr<MatchGameNamesTask> mOutstandingMatchGameNamesTask;
  std::string mNextGameNamesQuery;
  GetGameNameListCallback mNextGameNamesCallback;
};
