/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"

#include <memory>

namespace ttv {
namespace broadcast {
class IVideoCapture;
class IVideoEncoder;
class IVideoFrameQueue;
}  // namespace broadcast
}  // namespace ttv

/**
 * The interface for a component that can capture and provide video frames to the
 * broadcast system for encoding.
 *
 * The following methods should be called by the client.
 *   Initialize() should be called by the client prior to adding it to the system.
 *   Shutdown() should be called after removing from the system.
 *
 * The following methods will be called by the system during use.
 *   SetVideoEncoder() will be called prior to starting capture.
 *   SetFrameQueue() will be called prior to starting capture.
 *   Start() and Stop() will be called by the system when capture is to be started and stopped.
 */
class ttv::broadcast::IVideoCapture {
 public:
  virtual ~IVideoCapture() = default;

  /**
   * Retrieves the name for logging purposes.
   */
  virtual std::string GetName() const = 0;
  /**
   * Initializes the capturer.
   */
  virtual TTV_ErrorCode Initialize() = 0;
  /**
   * Synchronously stops frame submission and cleans up the internal pipeline.  The IVideoCapture should be disposable
   * after this returns.
   */
  virtual TTV_ErrorCode Shutdown() = 0;
  /**
   * Sets the IVideoEncoder that will be used to encode captured frames.
   */
  virtual TTV_ErrorCode SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) = 0;
  /**
   * Sets the IVideoFrameQueue instance to submit frames to once packaged by the IVideoEncoder.
   */
  virtual TTV_ErrorCode SetFrameQueue(const std::shared_ptr<IVideoFrameQueue>& queue) = 0;
  /**
   * Indicates that frame submission should begin.
   */
  virtual TTV_ErrorCode Start(const VideoParams& videoParams) = 0;
  /**
   * Indicates that frame submission should stop.  Any internal pipeline can be torn down.
   */
  virtual TTV_ErrorCode Stop() = 0;
};
