/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/broadcast/broadcasttypes.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"

#include <memory>
#include <vector>

namespace ttv {
namespace broadcast {
class IFrameWriter;
class VideoFrame;
class IVideoEncoder;
}  // namespace broadcast
}  // namespace ttv

/**
 * The interface for a video encoder.
 *
 * The following methods should be called by the client.
 *   Initialize() should be called by the client prior to adding it to the system.
 *   Shutdown() should be called after removing from the system.
 *
 * The following methods will be called by the system during use.
 *   SetFrameWriter() will be called prior to starting encoding.
 *   Start() and Stop() will be called by the system when capture is to be started and stopped.
 *   SubmitFrame() will be called periodically to kick off an encode of a frame.
 */
class ttv::broadcast::IVideoEncoder {
 public:
  virtual ~IVideoEncoder() = default;

 public:
  /**
   * Retrieves the name for logging purposes.
   */
  virtual std::string GetName() const = 0;
  /**
   * Determines whether or not the encoder supports changing the target bitrate on the fly.
   */
  virtual bool SupportsBitRateAdjustment() const = 0;
  /**
   * Initializes the encoder. The client is responsible for calling this before
   */
  virtual TTV_ErrorCode Initialize() = 0;
  virtual TTV_ErrorCode Shutdown() = 0;
  /**
   * Called by the system before encoding begins.
   */
  virtual TTV_ErrorCode SetFrameWriter(const std::shared_ptr<IFrameWriter>& frameWriter) = 0;
  /*
   * Called by the system to ensure the video parameters are valid for the encoder.
   * @return
   *    - TTV_EC_SUCCESS - Video parameters are valid for this encoder.
   *    - TTV_EC_BROADCAST_INVALID_RESOLUTION - Resolution is not valid for this encoder.
   *    - TTV_EC_BROADCAST_INVALID_FPS - Frames per second is not valid for this encoder.
   *    - TTV_EC_INVALID_ARG - Some other parameter is an invalid value.
   */
  virtual TTV_ErrorCode ValidateVideoParams(const VideoParams& videoParams) const = 0;
  /*
   * Called by the system when the stream starts.
   */
  virtual TTV_ErrorCode Start(uint32_t streamIndex, const VideoParams& videoParams) = 0;
  /**
   * This will be called periodically to kick off an encode of a frame.  This should not block to encode the frame
   * synchronously.
   */
  virtual TTV_ErrorCode SubmitFrame(const std::shared_ptr<VideoFrame>& videoFrame) = 0;
  /**
   * Called by the system when the stream is stopping.
   */
  virtual TTV_ErrorCode Stop() = 0;
  /**
   * Called by the system to get the Sps/Pps data from the encoder.
   */
  virtual TTV_ErrorCode GetSpsPps(std::vector<uint8_t>& sps, std::vector<uint8_t>& pps) = 0;
  /**
   * Called by the system to validate each frame as it is processed.
   */
  virtual TTV_ErrorCode ValidateFrame(const std::shared_ptr<VideoFrame>& videoframe) = 0;
  /**
   * Sets the desired bitrate in 1000s of bits per second.  This will only have an effect if SupportsBitrateAdjustment()
   * returns true. TTV_EC_UNSUPPORTED will be returned if not supported.
   */
  virtual TTV_ErrorCode SetTargetBitRate(uint32_t kbps) = 0;
  /**
   * Determines if the concrete implementation supports the given protocol.  The template parameter must be known to
   * derive from IVideoFrameReceiver.
   */
  template <typename T>
  bool SupportsReceiverProtocol() const {
    return SupportsReceiverProtocol(T::GetReceiverTypeId());
  }
  /**
   * Determines if the concrete implementation supports the given protocol.
   */
  virtual bool SupportsReceiverProtocol(IVideoFrameReceiver::ReceiverTypeId typeId) const = 0;
  /**
   * Obtains the receiver for the given protocol.  If not implemented, null will be returned.
   */
  virtual std::shared_ptr<IVideoFrameReceiver> GetReceiverImplementation(
    IVideoFrameReceiver::ReceiverTypeId typeId) = 0;
};
