#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/broadcast/broadcastapi.h"
#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/iingesttester.h"
#include "twitchsdk/broadcast/imuxer.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/testaudiocapturer.h"
#include "twitchsdk/broadcast/testvideocapturer.h"
#include "twitchsdk/broadcast/testvideodata.h"
#include "twitchsdk/core/assertion.h"

// TODO: We really should define platform-specific stuff differently
#if TTV_PLATFORM_WIN32
#include "twitchsdk/broadcast/intelvideoencoder.h"
#include "twitchsdk/broadcast/lameaudioencoder.h"
#include "twitchsdk/broadcast/testwindesktopvideocapturer.h"
#include "twitchsdk/broadcast/winaudiocapture.h"

#elif TTV_PLATFORM_DARWIN
#include "twitchsdk/broadcast/appleaacencoder.h"
#include "twitchsdk/broadcast/appleaudiocapture.h"
#include "twitchsdk/broadcast/applevideoencoder.h"
#include "twitchsdk/broadcast/darwintestcameracapturer.h"
#include "twitchsdk/broadcast/testmacdesktopvideocapturer.h"

#endif

#include <sstream>

using namespace ttv;
using namespace ttv::broadcast;

namespace ttv {
std::string ToString(BroadcastState state) {
  switch (state) {
    case BroadcastState::Initialized:
      return "Initialized";
    case BroadcastState::ReadyToBroadcast:
      return "ReadyToBroadcast";
    case BroadcastState::StartingBroadcast:
      return "StartingBroadcast";
    case BroadcastState::Broadcasting:
      return "Broadcasting";
    case BroadcastState::StoppingBroadcast:
      return "StoppingBroadcast";
    default:
      return "???";
  }
}

void Print(const IngestServer& server, std::string indent) {
  std::cout << indent << PRINT_FIELD(server, serverId) << std::endl;
  std::cout << indent << PRINT_FIELD(server, serverName) << std::endl;
  std::cout << indent << PRINT_FIELD(server, serverUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(server, priority) << std::endl;
}

void Print(const std::vector<IngestServer>& list, std::string indent) {
  std::cout << indent << "IngestServerList (" << list.size() << ")" << std::endl;
  for (uint32_t i = 0; i < list.size(); ++i) {
    const auto& server = list[i];
    std::cout << indent << i << ": " << std::endl;
    Print(server, indent + "    ");
  }
}

std::string ToString(IIngestTester::TestState state) {
  switch (state) {
    case IIngestTester::TestState::Stopped:
      return "Stopped";
    case IIngestTester::TestState::Connecting:
      return "Connecting";
    case IIngestTester::TestState::Testing:
      return "Testing";
    case IIngestTester::TestState::Disconnecting:
      return "Disconnecting";
    case IIngestTester::TestState::Finished:
      return "Finished";
    case IIngestTester::TestState::Failed:
      return "Failed";
    default:
      return "???";
  }
}
}  // namespace ttv

namespace {
const char* kTraceComponents[] = {
  "rtmp", "RtmpState", "ChannelInfoTask", "GameStreamsTask", "IngestListTask", "MatchGameNamesTask",
  "RecordingStatusTask", "RunCommercialTask", "SetStreamInfoTask", "StreamInfoTask", "AudioMixer", "AudioStreamer",
  "FrameWriter",
  //"Streamer",
  "TwitchAPI",
  //"VideoFrameQueue",
  //"VideoStreamer",
  //"IntelBaseAllocator",
  //"IntelD3DAllocator",
  //"IntelSysAllocator",
  "IntelVideoEncoder",
  //"LameAudioEncoder",
  "FlvMuxer", "AppleAacEncoder", "AppleAudioCapture", "WinAudioCapture", "AppleVideoEncoder",
  //"IngestTester",
  //"TestVideoCapturer",
  //"TestAudioCapturer"
};

class BroadcastApiListener;
class IngestTesterListener;

std::shared_ptr<BroadcastAPI> gBroadcastApi;
VideoParams gVideoParams;
std::shared_ptr<BroadcastApiListener> gBroadcastApiListener;
std::shared_ptr<IngestTesterListener> gIngestTesterListener;
std::shared_ptr<IVideoEncoder> gVideoEncoder;
std::shared_ptr<IVideoEncoder> gIngestTesterVideoEncoder;
std::shared_ptr<IAudioEncoder> gAudioEncoder;
std::shared_ptr<IVideoCapture> gVideoCapturer;
std::shared_ptr<IMuxer> gCustomMuxer;
std::map<std::string, std::shared_ptr<IAudioCapture>>
  gAudioCapturers;  // Mapping of layer id to the capturer for the layer
std::vector<IngestServer> gIngestServerList;
std::map<UserId, std::shared_ptr<IIngestTester>> gIngestTesterMap;
std::hash<std::string> gStringHasher;

struct IngestTestAllData {
  std::shared_ptr<IVideoEncoder> videoEncoder;
  std::shared_ptr<IIngestTester> ingestTester;
  std::shared_ptr<IngestTesterListenerProxy> ingestTesterListener;
  std::shared_ptr<std::vector<IngestServer>> ingestList;
};
std::shared_ptr<IngestTestAllData> gIngestTestAllData;

TTV_ErrorCode CreateVideoEncoder(std::shared_ptr<IVideoEncoder>& result) {
#if TTV_PLATFORM_WIN32
  result = std::make_shared<IntelVideoEncoder>();
#elif TTV_PLATFORM_DARWIN
  result = std::make_shared<AppleVideoEncoder>();
#endif
  if (result != nullptr) {
    TTV_ErrorCode ec = result->Initialize();
    TTV_ASSERT(TTV_SUCCEEDED(ec));
    return ec;
  } else {
    return TTV_EC_NOT_AVAILABLE;
  }
}

void PrintAvailableAudioCapturers() {
  std::cout << "Available audio capturers:" << std::endl;

  for (const auto& kvp : gAudioCapturers) {
    std::cout << "  " << kvp.first << std::endl;
  }
}

std::shared_ptr<IIngestTester> GetIngestTester(UserId userId) {
  auto iter = gIngestTesterMap.find(userId);
  if (iter == gIngestTesterMap.end()) {
    std::cout << "Ingest tester doesn't exist for user" << std::endl;
    return nullptr;
  } else {
    return iter->second;
  }
}

IngestServer GetIngestServer(uint32_t serverId) {
  auto serverIter = std::find_if(gIngestServerList.begin(), gIngestServerList.end(),
    [serverId](const IngestServer& server) { return server.serverId == serverId; });

  return (serverIter != gIngestServerList.end()) ? *serverIter : IngestServer();
}

IngestServer GetIngestServer(const std::string& name) {
  auto serverIter = std::find_if(gIngestServerList.begin(), gIngestServerList.end(),
    [&name](const IngestServer& server) { return server.serverName == name; });

  return (serverIter != gIngestServerList.end()) ? *serverIter : IngestServer();
}

class BroadcastApiListener : public ttv::broadcast::IBroadcastAPIListener {
 public:
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) override {
    std::cout << "ModuleStateChanged: " << ::ToString(state) << " " << ErrorToString(ec) << std::endl;
  }

  virtual void BroadcastStateChanged(TTV_ErrorCode ec, BroadcastState state) override {
    std::cout << "BroadcastStateChanged: " << ErrorToString(ec) << " " << ToString(state) << std::endl;
  }

  virtual void BroadcastBandwidthWarning(TTV_ErrorCode ec, uint32_t backupMilliseconds) override {
    std::cout << "BroadcastBandwidthWarning: " << ErrorToString(ec) << " " << backupMilliseconds << std::endl;
  }

  virtual void BroadcastFrameSubmissionIssue(TTV_ErrorCode ec) override {
    std::cout << "BroadcastFrameSubmissionIssue: " << ErrorToString(ec) << std::endl;
  }

  virtual void StreamInfoFetched(TTV_ErrorCode ec, const StreamInfo& streamInfo) override {
    std::cout << "StreamInfoFetched: " << ErrorToString(ec) << std::endl;
    Print(streamInfo, "  ");
  }

  virtual void StreamKeyError(const CanTheyError& error) override {
    std::cout << "StreamKeyError: " << std::endl;
    Print(error, "  ");
  }
};

class IngestTesterListener : public IIngestTesterListener {
 public:
  virtual void BroadcastIngestTesterStateChanged(IIngestTester* source) override {
    UserId userId = 0;
    source->GetUserId(userId);

    IIngestTester::TestState state;
    source->GetTestState(state);

    std::cout << "BroadcastIngestTestStateChanged: [UserId: " << userId << "]  " << ToString(state) << std::endl;

    if (state == IIngestTester::TestState::Finished) {
      IngestServer server;
      source->GetIngestServer(server);

      uint32_t kbps = 0;
      source->GetMeasuredKbps(kbps);

      std::cout << "  " << server.serverName << " " << kbps << " kbps" << std::endl;
    } else if (state == IIngestTester::TestState::Failed) {
      IngestServer server;
      source->GetIngestServer(server);

      TTV_ErrorCode ec = TTV_EC_SUCCESS;
      source->GetTestError(ec);

      std::cout << "  " << server.serverName << " " << ErrorToString(ec) << std::endl;
    }
  }
};

class CustomMuxer : public IMuxer {
 public:
  virtual TTV_ErrorCode Start(const MuxerParameters& parameters) override {
    std::cout << "CustomMuxer::Start()" << std::endl;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode WriteVideoPacket(const Packet& packet) override {
    std::cout << "CustomMuxer::WriteVideoPacket()" << std::endl;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode WriteAudioPacket(const Packet& packet) override {
    std::cout << "CustomMuxer::WriteAudioPacket()" << std::endl;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode WriteVideoSpsPps(const std::vector<uint8_t>& sps, const std::vector<uint8_t>& pps) override {
    std::cout << "CustomMuxer::WriteVideoSpsPps()" << std::endl;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode Stop() override {
    std::cout << "CustomMuxer::Stop()" << std::endl;

    return TTV_EC_SUCCESS;
  }
};

std::shared_ptr<CommandCategory> RegisterBroadcastCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Broadcast", "The ttv::broadcast::BroadcastAPI command interface");

  category->AddCommand("BroadcastInit")
    .AddFunction()
    .Description("Calls BroadcastAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      gBroadcastApi->SetListener(gBroadcastApiListener);

      TTV_ErrorCode ec = gBroadcastApi->Initialize([](TTV_ErrorCode ec) {
        std::cout << "BroadcastInit completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastGetState")
    .AddFunction()
    .Description("Calls BroadcastAPI::GetState")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      IModule::State state = gBroadcastApi->GetState();
      std::cout << "  " << ::ToString(state) << std::endl;
      return true;
    });

  category->AddCommand("BroadcastShutdown")
    .AddFunction()
    .Description("Calls BroadcastAPI::Shutdown")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "BroadcastShutdown completed with error:";
        ReportCommandResult(ec);
      });

      if (TTV_SUCCEEDED(ec)) {
        gIngestServerList.clear();
      }

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetActiveUser")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetActiveUser")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetActiveUser(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetBroadcasterSoftware")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetBroadcasterSoftware")
    .AddParam("str", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetBroadcasterSoftware(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastGetBroadcastState")
    .AddFunction()
    .Description("Calls BroadcastAPI::GetBroadcastState")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      BroadcastState state = BroadcastState::Initialized;

      TTV_ErrorCode ec = gBroadcastApi->GetBroadcastState(state);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  " << ToString(state) << std::endl;
      }

      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetStreamInfo")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetStreamInfo")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .AddParam("game", ParamType::String)
    .AddParam("title", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetStreamInfo(params[0], params[1], params[2], params[3],
        [](TTV_ErrorCode ec) { std::cout << "BroadcastSetStreamInfoComplete: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastRunCommercial")
    .AddFunction()
    .Description("Calls BroadcastAPI::RunCommercial")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelName", ParamType::String)
    .AddParam("timeBreakSeconds", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->RunCommercial(params[0], params[1], params[2],
        [](TTV_ErrorCode ec) { std::cout << "BroadcastRunCommercialComplete: " << ErrorToString(ec) << std::endl; });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetVideoParams")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetVideoParams")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetVideoParams(gVideoParams);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastConfigureForBandwidth")
    .AddFunction()
    .Description("Calls VideoParams::ConfigureForBandwidth")
    .AddParam("expectedKbps", ParamType::UInt32)
    .AddParam("frameRate", ParamType::UInt32)
    .AddParam("bitsPerPixel", ParamType::Float)
    .AddParam("aspectRatio", ParamType::Float)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      gVideoParams.automaticBitRateAdjustmentEnabled = false;
      TTV_ErrorCode ec = VideoParams::ConfigureForBandwidth(params[0], params[1], params[2], params[3], gVideoParams);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastConfigureForResolution")
    .AddFunction()
    .Description("Calls VideoParams::ConfigureForResolution")
    .AddParam("width", ParamType::UInt32)
    .AddParam("height", ParamType::UInt32)
    .AddParam("frameRate", ParamType::UInt32)
    .AddParam("bitsPerPixel", ParamType::Float)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = VideoParams::ConfigureForResolution(params[0], params[1], params[2], params[3], gVideoParams);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastGetCurrentBroadcastTime")
    .AddFunction()
    .Description("Calls BroadcastAPI::GetCurrentBroadcastTime")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      uint64_t result = 0;
      TTV_ErrorCode ec = gBroadcastApi->GetCurrentBroadcastTime(result);
      if (TTV_SUCCEEDED(ec)) {
        // TODO: Maybe print as a nice time string
        std::cout << "  " << result << std::endl;
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetAudioLayerEnabled")
    .AddFunction()
    .Description("Calls BroadcastAPI::BroadcastSetAudioLayerEnabled")
    .AddParam("layerId", ParamType::UInt32)
    .AddParam("enabled", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetAudioLayerEnabled(params[0], params[1]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetAudioLayerVolume")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetAudioLayerVolume")
    .AddParam("layerId", ParamType::UInt32)
    .AddParam("volume", ParamType::Float)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetAudioLayerVolume(params[0], params[1]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetAudioLayerMuted")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetAudioLayerMuted")
    .AddParam("layerId", ParamType::UInt32)
    .AddParam("bool", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetAudioLayerMuted(params[0], params[1]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastFetchIngestServerList")
    .AddFunction()
    .Description("Calls BroadcastAPI::FetchIngestServerList")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec =
        gBroadcastApi->FetchIngestServerList([](TTV_ErrorCode ec, const std::vector<IngestServer>& result) {
          gIngestServerList = result;

          std::cout << "BroadcastIngestServerListFetchComplete: " << ErrorToString(ec) << std::endl;

          if (TTV_SUCCEEDED(ec)) {
            Print(gIngestServerList, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastGetSelectedIngestServer")
    .AddFunction()
    .Description("Calls BroadcastAPI::GetSelectedIngestServer")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      IngestServer result;

      TTV_ErrorCode ec = gBroadcastApi->GetSelectedIngestServer(result);
      if (TTV_SUCCEEDED(ec)) {
        Print(result, "  ");
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetSelectedIngestServer")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetSelectedIngestServer")
    .AddParam("index", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      if (gIngestServerList.empty()) {
        std::cout << "Servers not available" << std::endl;
        return false;
      }

      IngestServer server;

      uint32_t index = params[0].GetUInt32();
      if (index >= gIngestServerList.size()) {
        return false;
      }

      server = gIngestServerList[index];

      TTV_ErrorCode ec = gBroadcastApi->SetSelectedIngestServer(server);
      ReportCommandResult(ec);
      return true;
    })
    .Done()
    .AddFunction()
    .Description("Calls BroadcastAPI::SetSelectedIngestServer")
    .AddParam("name", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      if (gIngestServerList.empty()) {
        std::cout << "Servers not available" << std::endl;
        return false;
      }

      IngestServer selectedServer = GetIngestServer(params[0].GetString());

      TTV_ErrorCode ec = gBroadcastApi->SetSelectedIngestServer(selectedServer);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetCustomIngestServer")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetSelectedIngestServer with a custom server")
    .AddParam("name", ParamType::String)
    .AddParam("url", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      IngestServer server;

      server.serverName = params[0].GetString();
      server.serverUrl = params[1].GetString();

      TTV_ErrorCode ec = gBroadcastApi->SetSelectedIngestServer(server);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetVideoEncoder")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetVideoEncoder")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetVideoEncoder(gVideoEncoder);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetAudioEncoder")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetAudioEncoder")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetAudioEncoder(gAudioEncoder);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetVideoCapturer")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetVideoCapturer")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetVideoCapturer(gVideoCapturer);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetAudioCapturer")
    .AddFunction()
    .AddParam("layerName", ParamType::String)
    .Description(
      "Calls BroadcastAPI::SetAudioCapturer with a valid instance, using a hash of layerName as the layer id.")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gAudioCapturers.find(params[0]);
      if (iter != gAudioCapturers.end()) {
        AudioLayerId layerId = static_cast<AudioLayerId>(gStringHasher(params[0].GetString()));
        TTV_ErrorCode ec = gBroadcastApi->SetAudioCapturer(layerId, iter->second);
        ReportCommandResult(ec);
        return true;
      } else {
        std::cout << "Audio layer id not valid: " << params[1].GetString() << std::endl;
        PrintAvailableAudioCapturers();
        return false;
      }
    });

  category->AddCommand("BroadcastListAvailableAudioCapturers")
    .AddFunction()
    .Description("Lists the audio capturers that the sample has available")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      PrintAvailableAudioCapturers();

      return true;
    });

  category->AddCommand("BroadcastRemoveAudioCapturer")
    .AddFunction()
    .AddParam("layerName", ParamType::String)
    .Description("Calls BroadcastAPI::SetAudioCapturer with a null instance to clear the capturer")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      AudioLayerId layerId = static_cast<AudioLayerId>(gStringHasher(params[0].GetString()));

      TTV_ErrorCode ec = gBroadcastApi->SetAudioCapturer(layerId, nullptr);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastStart")
    .AddFunction()
    .Description("Calls BroadcastAPI::Start")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->StartBroadcast([](TTV_ErrorCode /*ec*/) {});
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetOutputPath")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetOutputPath")
    .AddParam("path", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      const std::string& path = ttv::ContainerizedPath(params[0]);
      TTV_ErrorCode ec = gBroadcastApi->SetOutputPath(std::wstring(path.begin(), path.end()));
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("BroadcastSetConnectionType")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetConnectionType")
    .AddParam("connectionType", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      ConnectionType connectionType;
      if (params[0] == "unknown") {
        connectionType = ConnectionType::Unknown;
      } else if (params[0] == "wifi") {
        connectionType = ConnectionType::Wifi;
      } else if (params[0] == "cellular") {
        connectionType = ConnectionType::Cellular;
      } else if (params[0] == "ethernet") {
        connectionType = ConnectionType::Ethernet;
      } else {
        std::cout << "Unrecognized connection type: " << params[0].GetString() << std::endl;
        return false;
      }

      TTV_ErrorCode ec = gBroadcastApi->SetConnectionType(connectionType);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastSetSessionId")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetSessionId")
    .AddParam("sessionId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetSessionId(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastStop")
    .AddFunction()
    .Description("Calls BroadcastAPI::Stop")
    .AddParam("reason", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->StopBroadcast(params[0], [](TTV_ErrorCode /*ec*/) {});
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastAddCustomMuxer")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetCustomMuxer with a test muxer")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetCustomMuxer(gCustomMuxer);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastRemoveCustomMuxer")
    .AddFunction()
    .Description("Calls BroadcastAPI::SetCustomMuxer with nullptr")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gBroadcastApi->SetCustomMuxer(nullptr);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastCreateIngestTester")
    .AddFunction()
    .Description("Calls BroadcastAPI::CreateIngestTester")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gIngestTesterMap.find(params[0]);
      if (iter != gIngestTesterMap.end()) {
        std::cout << "Ingest tester already created for user" << std::endl;
        return true;
      }

      std::shared_ptr<IIngestTester> ingestTester;

      TTV_ErrorCode ec = gBroadcastApi->CreateIngestTester(
        params[0], gIngestTesterListener, kTestVideoIngestData, kTestVideoIngestDataLength, ingestTester);
      if (TTV_SUCCEEDED(ec)) {
        gIngestTesterMap[params[0]] = ingestTester;
      }
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastDisposeIngestTester")
    .AddFunction()
    .Description("Calls BroadcastAPI::DisposeIngestTester")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto iter = gIngestTesterMap.find(params[0]);
      if (iter == gIngestTesterMap.end()) {
        std::cout << "Ingest tester doesn't exist for user" << std::endl;
        return true;
      }

      auto ingestTester = iter->second;
      gIngestTesterMap.erase(iter);

      TTV_ErrorCode ec = gBroadcastApi->DisposeIngestTester(ingestTester);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("BroadcastIngestTesterStart")
    .AddFunction()
    .Description("Calls IIngestTester::Start")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("serverId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto ingestTester = GetIngestTester(params[0]);
      if (ingestTester != nullptr) {
        IngestServer selectedServer = GetIngestServer(params[1].GetUInt32());

        TTV_ErrorCode ec = ingestTester->Start(selectedServer);
        ReportCommandResult(ec);
      }

      return true;
    });

  category->AddCommand("BroadcastIngestTesterCancel")
    .AddFunction()
    .Description("Calls IIngestTester::Cancel")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto ingestTester = GetIngestTester(params[0]);
      if (ingestTester != nullptr) {
        TTV_ErrorCode ec = ingestTester->Cancel();
        ReportCommandResult(ec);
      }

      return true;
    });

  category->AddCommand("BroadcastIngestTesterSetTestDurationMilliseconds")
    .AddFunction()
    .Description("Calls IIngestTester::SetTestDurationMilliseconds")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("duration", ParamType::UInt64)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      auto ingestTester = GetIngestTester(params[0]);
      if (ingestTester != nullptr) {
        TTV_ErrorCode ec = ingestTester->SetTestDurationMilliseconds(params[1]);
        ReportCommandResult(ec);
      }

      return true;
    });

  category->AddCommand("BroadcastIngestTesterTestAll")
    .AddFunction()
    .Description("Runs the ingest tester on all servers")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("duration", ParamType::UInt64)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      if (gIngestTestAllData != nullptr) {
        std::cout << "Ingest test all already in progress" << std::endl;
        return false;
      } else if (gIngestServerList.empty()) {
        std::cout << "Ingest server list not yet available" << std::endl;
        return false;
      }

      gIngestTestAllData = std::make_shared<IngestTestAllData>();

      CreateVideoEncoder(gIngestTestAllData->videoEncoder);

      gIngestTestAllData->ingestList = std::make_shared<std::vector<IngestServer>>();
      (*gIngestTestAllData->ingestList) = gIngestServerList;

      auto done = []() {
        std::cout << "Done testing all servers" << std::endl;

        gIngestTestAllData.reset();
      };

      auto pump = [done]() {
        IIngestTester::TestState state;
        gIngestTestAllData->ingestTester->GetTestState(state);

        if (state == IIngestTester::TestState::Finished) {
          IngestServer server;
          gIngestTestAllData->ingestTester->GetIngestServer(server);

          uint32_t kbps = 0;
          gIngestTestAllData->ingestTester->GetMeasuredKbps(kbps);

          std::cout << server.serverName << " " << kbps << " kbps" << std::endl;
        } else if (state == IIngestTester::TestState::Failed) {
          IngestServer server;
          gIngestTestAllData->ingestTester->GetIngestServer(server);

          TTV_ErrorCode ec = TTV_EC_SUCCESS;
          gIngestTestAllData->ingestTester->GetTestError(ec);

          std::cout << server.serverName << " " << ErrorToString(ec) << std::endl;
        }

        if (state == IIngestTester::TestState::Finished || state == IIngestTester::TestState::Failed ||
            state == IIngestTester::TestState::Stopped) {
          if (!gIngestTestAllData->ingestList->empty()) {
            IngestServer server;
            server = (*gIngestTestAllData->ingestList)[0];
            gIngestTestAllData->ingestList->erase(gIngestTestAllData->ingestList->begin());

            gIngestTestAllData->ingestTester->Start(server);
          } else {
            done();
          }
        }
      };

      gIngestTestAllData->ingestTesterListener = std::make_shared<IngestTesterListenerProxy>();
      gIngestTestAllData->ingestTesterListener->mOnStateChangedFunc = [pump](IIngestTester* /*source*/) { pump(); };

      TTV_ErrorCode ec = gBroadcastApi->CreateIngestTester(params[0], gIngestTestAllData->ingestTesterListener,
        kTestVideoIngestData, kTestVideoIngestDataLength, gIngestTestAllData->ingestTester);
      ec = gIngestTestAllData->ingestTester->SetTestDurationMilliseconds(params[1]);

      pump();

      return true;
    });

  category->AddCommand("PrintJavaBroadcastErrorCodes")
    .AddFunction()
    .Description("Dumps the Java snippet for updated values in BroadcastErrorCode.java")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::vector<EnumValue> values;
      ttv::broadcast::GetBroadcastErrorCodeValues(values);

      for (const auto& v : values) {
        std::cout << "public static final ErrorCode " << v.name << " = new BroadcastErrorCode(0x" << std::hex << v.value
                  << ", \"" << v.name << "\");" << std::endl;
      }

      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));

      gBroadcastApi = std::make_shared<BroadcastAPI>();
      gBroadcastApi->SetCoreApi(coreApi);
      context->RegisterModule(gBroadcastApi);

      gBroadcastApiListener = std::make_shared<BroadcastApiListener>();
      gIngestTesterListener = std::make_shared<IngestTesterListener>();

      gCustomMuxer = std::make_shared<CustomMuxer>();

      // TODO: Allow the cli to change the generators and encoders
      CreateVideoEncoder(gVideoEncoder);
      CreateVideoEncoder(gIngestTesterVideoEncoder);

#if TTV_PLATFORM_WIN32
      gAudioEncoder = std::make_shared<LameAudioEncoder>();
#elif TTV_PLATFORM_DARWIN
      gAudioEncoder = std::make_shared<AppleAacEncoder>();
#endif
      if (gAudioEncoder != nullptr) {
        TTV_ErrorCode ec = gAudioEncoder->Initialize();
        TTV_ASSERT(TTV_SUCCEEDED(ec));
      }

#if TTV_PLATFORM_WIN32
      gVideoCapturer = std::make_shared<test::TestWinDesktopVideoCapturer>();
#elif TTV_PLATFORM_DARWIN
#if TTV_TARGET_IOS
      gVideoCapturer =
        std::make_shared<test::DarwinTestCameraVideoCapturer>(test::DarwinTestCameraVideoCapturer::CameraType::Front);
#else
      gVideoCapturer = std::make_shared<test::TestVideoCapturer>();
#endif
#else
      gVideoCapturer = std::make_shared<test::TestVideoCapturer>();
#endif
      if (gVideoCapturer != nullptr) {
        TTV_ErrorCode ec = gVideoCapturer->Initialize();
        TTV_ASSERT(TTV_SUCCEEDED(ec));
      }

      gAudioCapturers["test"] = std::make_shared<test::TestAudioCapturer>();
      gAudioCapturers["test"]->Initialize();

#if TTV_PLATFORM_WIN32
      gAudioCapturers["system"] = std::make_shared<WinAudioCapture>(WinAudioCapture::CaptureType::System);
      gAudioCapturers["microphone"] = std::make_shared<WinAudioCapture>(WinAudioCapture::CaptureType::Microphone);
#elif TTV_PLATFORM_DARWIN
      gAudioCapturers["system"] = std::make_shared<AppleAudioCapture>(AppleAudioCapture::CaptureType::System);
      gAudioCapturers["microphone"] = std::make_shared<AppleAudioCapture>(AppleAudioCapture::CaptureType::Microphone);
#endif

      if (gAudioCapturers["system"] != nullptr) {
        gAudioCapturers["system"]->Initialize();
      }

      if (gAudioCapturers["microphone"] != nullptr) {
        gAudioCapturers["microphone"]->Initialize();
      }

      auto commands = RegisterBroadcastCommands(context);
      context->RegisterCommandCategory(commands);
    },
    1);
});
}  // namespace
