/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/testaudiocapturer.h"

#include "twitchsdk/broadcast/audioconstants.h"
#include "twitchsdk/broadcast/iaudiomixer.h"
#include "twitchsdk/broadcast/ipcmaudioframereceiver.h"
#include "twitchsdk/core/memory.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

#include <math.h>

namespace {
using namespace ttv::broadcast;

const float kPi = 3.14159f;
const float kTwoPi = kPi * 2.0f;
const uint32_t kNumAudioChannels = 2;
const uint32_t kProcessFPS = 30;  // Audio return FPS
const uint32_t kMillisecondsBetweenProcesses = 1000 / kProcessFPS;
const size_t kNumSamplesPerChannel = kAudioEncodeRate / kProcessFPS;
const size_t kNumSamplesPerChunk = kNumAudioChannels * kNumSamplesPerChannel;
const float kPitchBase = 440.0f;  // Hz, perfect A
const float kPitchFrequency = 5;
const float kPitchStep = kTwoPi / static_cast<float>(kAudioEncodeRate) / kPitchFrequency;
}  // namespace

static_assert(ttv::broadcast::kAudioEncodeRate == 44100, "kAudioEncodeRate needs to be 44100");

ttv::broadcast::test::TestAudioCapturer::TestAudioCapturer()
    : mSampleTime(0), mLastProcessTime(0), mWaveFormProgress(0), mPitchProgress(0) {}

ttv::broadcast::test::TestAudioCapturer::~TestAudioCapturer() {
  Shutdown();
}

std::string ttv::broadcast::test::TestAudioCapturer::GetName() const {
  return "TestAudioCapturer";
}

TTV_ErrorCode ttv::broadcast::test::TestAudioCapturer::Initialize() {
  mBuffer.resize(kNumSamplesPerChunk);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::test::TestAudioCapturer::Shutdown() {
  mBuffer.resize(0);

  return Stop();
}

uint32_t ttv::broadcast::test::TestAudioCapturer::GetNumChannels() const {
  return kNumAudioChannels;
}

TTV_ErrorCode ttv::broadcast::test::TestAudioCapturer::Start() {
  TTV_ErrorCode ec = AudioCaptureBase::Start();

  if (TTV_SUCCEEDED(ec)) {
    mBuffer.resize(kNumSamplesPerChunk);

    std::shared_ptr<IAudioFrameReceiver> receiver =
      mAudioMixer->GetReceiverImplementation(IPcmAudioFrameReceiver::GetReceiverTypeId());

    if (receiver != nullptr) {
      mReceiver = std::static_pointer_cast<IPcmAudioFrameReceiver>(receiver);
    } else {
      ec = TTV_EC_BROADCAST_INVALID_ENCODER;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::test::TestAudioCapturer::Process(
  const std::shared_ptr<IAudioMixer>& mixer, uint64_t& lastSampleTime) {
  lastSampleTime = mSampleTime;

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // Make sure we submit audio at a reasonable rate
  uint64_t now = GetSystemClockTime();
  uint64_t delta = now - mLastProcessTime;

  if (SystemTimeToMs(delta) >= kMillisecondsBetweenProcesses) {
    mLastProcessTime = now;

    for (size_t i = 0; i < kNumSamplesPerChunk; i += kNumAudioChannels) {
      float pitch = kPitchBase + 100.0f * std::sin(mPitchProgress);
      float waveFormStep = kTwoPi * pitch / static_cast<float>(kAudioEncodeRate);
      float sample = std::sin(mWaveFormProgress) * 32767.0f;

      // Populate both channels with the same value
      mBuffer[i] = static_cast<int16_t>(sample);
      mBuffer[i + 1] = mBuffer[i];

      mWaveFormProgress += waveFormStep;
      mPitchProgress += kPitchStep;

      if (mWaveFormProgress >= kTwoPi) {
        mWaveFormProgress -= kTwoPi;
      }
      if (mPitchProgress >= kTwoPi) {
        mPitchProgress -= kTwoPi;
      }
    }

    // Submit for processing
    std::shared_ptr<AudioFrame> audioFrame;
    ec = mReceiver->PackageFrame(reinterpret_cast<const uint8_t*>(mBuffer.data()), kNumSamplesPerChannel,
      kNumAudioChannels, true, AudioSampleFormat::TTV_ASF_PCM_S16, 0, audioFrame);

    if (TTV_SUCCEEDED(ec)) {
      mixer->SubmitFrame(mAudioLayer, audioFrame);
    }

    mSampleTime += static_cast<uint64_t>(kNumSamplesPerChannel);
  }

  return ec;
}
