/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/testvideocapturer.h"

#include "twitchsdk/broadcast/irawvideoframereceiver.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"
#include "twitchsdk/core/memory.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

#define CREATE_RANDOM_FRAMES 0

namespace {
const char* kLoggerName = "TestVideoCapturer";
/**
 * Triple buffering is needed to prevent stalls.  One frame being encoded, one pending to be encoded and
 * another being filled with the current frame.  A fourth wouldn't hurt to ensure latency in handing off
 * buffers is minimized but isn't required if latency is low.
 */
const size_t kNumBuffers = 3;
}  // namespace

ttv::broadcast::test::TestVideoCapturer::TestVideoCapturer() : mColor1(0), mColor2(0), mTime(0) {
  mSubmitting = false;
  mShutdown = false;
}

ttv::broadcast::test::TestVideoCapturer::~TestVideoCapturer() {
  Shutdown();
}

std::string ttv::broadcast::test::TestVideoCapturer::GetName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) {
  if (mSubmitting) {
    return TTV_EC_INVALID_STATE;
  }

  mVideoEncoder = encoder;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::SetFrameQueue(const std::shared_ptr<IVideoFrameQueue>& queue) {
  if (mSubmitting) {
    return TTV_EC_INVALID_STATE;
  }

  mFrameQueue = queue;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::Initialize() {
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::Start(const VideoParams& videoParams) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "TestVideoCapturer::Start()");

  if (mSubmitting) {
    return TTV_EC_INVALID_STATE;
  }

  if (mVideoEncoder == nullptr || mFrameQueue == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  // We require a IRawVideoFrameReceiver to submit to
  if (!mVideoEncoder->SupportsReceiverProtocol(IRawVideoFrameReceiver::GetReceiverTypeId())) {
    return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
  }

  // Extract the IRawVideoFrameReceiver
  auto receiver = mVideoEncoder->GetReceiverImplementation(IRawVideoFrameReceiver::GetReceiverTypeId());
  TTV_ASSERT(receiver != nullptr);
  mRawVideoFrameReceiver = std::static_pointer_cast<IRawVideoFrameReceiver>(receiver);

  // Create buffers of the needed size
  for (size_t i = 0; i < kNumBuffers; ++i) {
    uint8_t* buffer = static_cast<uint8_t*>(AlignedAlloc(videoParams.outputHeight * videoParams.outputWidth * 4, 16));
    mAllBuffers.push_back(buffer);
    mFreeBuffers.push_back(buffer);
  }

  mVideoParams = videoParams;
  mColor1 = 0xFF0000FF;
  mColor2 = 0x0000FFFF;
  mTime = 0;

  auto generateFrameFunc = [this]() {
    auto unlockCallback = [this](
                            const uint8_t* frameBuffer) { mFreeBuffers.push_back(const_cast<uint8_t*>(frameBuffer)); };

    uint64_t sleepDuration = 1000 / mVideoParams.targetFramesPerSecond;

    while (!mShutdown) {
      // ttv::trace::Message(kLoggerName, MessageLevel::Debug, "Tick");

      uint64_t start = GetSystemTimeMilliseconds();

      if (mFreeBuffers.empty()) {
        ttv::trace::Message(kLoggerName, MessageLevel::Debug, "no free buffers");
      } else {
        // Get the next free frame
        uint8_t* buffer = mFreeBuffers[0];
        mFreeBuffers.erase(mFreeBuffers.begin());

        // Fill the frame
        PixelFormat pixelFormat = PixelFormat::TTV_PF_BGRA;
        bool flip = false;
        GenerateFrame(buffer, pixelFormat, flip);

        // Package the frame
        std::shared_ptr<VideoFrame> videoFrame;
        TTV_ErrorCode ec = mRawVideoFrameReceiver->PackageFrame(
          buffer, pixelFormat, flip, GetSystemClockTime(), unlockCallback, videoFrame);
        TTV_ASSERT(TTV_SUCCEEDED(ec));
        TTV_ASSERT(videoFrame != nullptr);

        // Submit for processing
        ec = mFrameQueue->AddVideoFrame(videoFrame);
        if (TTV_FAILED(ec)) {
          mShutdown = true;
        }
      }

      uint64_t elapsedTime = GetSystemTimeMilliseconds() - start;
      if (elapsedTime < sleepDuration) {
        Sleep(sleepDuration - elapsedTime);
      } else {
        ttv::trace::Message(kLoggerName, MessageLevel::Debug, "Frame generation taking too long.");
      }
    }

    // Thread shutting down now, wait for all buffers to be released
    while (mFreeBuffers.size() < mAllBuffers.size()) {
      Sleep(10);
    }

    // Clean up buffers
    for (size_t i = 0; i < kNumBuffers; ++i) {
      AlignedFree(mAllBuffers[i]);
    }

    mAllBuffers.clear();
    mFreeBuffers.clear();

    mSubmitting = false;
    mShutdown = false;
  };

  TTV_ErrorCode ec = CreateThread(generateFrameFunc, "ttv::broadcast::test::TestVideoCapturer", mThread);

  if (TTV_SUCCEEDED(ec)) {
    mSubmitting = true;
    mShutdown = false;

    mThread->Run();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::Stop() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "TestVideoCapturer::Stop()");

  mShutdown = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::test::TestVideoCapturer::Shutdown() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "TestVideoCapturer::Shutdown()");

  mShutdown = true;

  if (mThread != nullptr) {
    mThread->Join();
    mThread.reset();
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::test::TestVideoCapturer::GenerateFrame(
  uint8_t* buffer, PixelFormat& pixelFormat, bool& requiresVerticalFlip) {
  pixelFormat = PixelFormat::TTV_PF_BGRA;
  requiresVerticalFlip = false;

#if CREATE_RANDOM_FRAMES
  uint32_t* arr = reinterpret_cast<uint32_t*>(buffer);
  uint32_t size = mVideoParams.outputWidth * mVideoParams.outputHeight;
  for (uint32_t i = 0; i < size; ++i) {
    arr[i] = rand();
  }
#else
  // NOTE: This code is incredibly slow

  // Populate it with solid color
  uint32_t c1 = mColor1;
  uint32_t c2 = mColor2;
  uint8_t* b1 = reinterpret_cast<uint8_t*>(&c1);
  uint8_t* b2 = reinterpret_cast<uint8_t*>(&c2);

  float r = static_cast<float>(b1[0]) * (1.0f - mTime) + static_cast<float>(b2[0]) * mTime;
  float g = static_cast<float>(b1[1]) * (1.0f - mTime) + static_cast<float>(b2[1]) * mTime;
  float b = static_cast<float>(b1[2]) * (1.0f - mTime) + static_cast<float>(b2[2]) * mTime;

  uint32_t clr = static_cast<uint32_t>(
    ((static_cast<uint8_t>(r) << 24) | (static_cast<uint8_t>(g) << 16) | (static_cast<uint8_t>(b) << 8) | 0xFF));
  uint32_t size = mVideoParams.outputHeight * mVideoParams.outputWidth;

  uint32_t* arr = reinterpret_cast<uint32_t*>(buffer);
  for (uint32_t i = 0; i < size; ++i) {
    arr[i] = clr;
  }

  mTime += 1.0f / static_cast<float>(mVideoParams.targetFramesPerSecond);

  // Choose the next color to transition to
  if (mTime >= 1.0f) {
    mTime -= 1.0f;

    std::swap(mColor1, mColor2);
  }
#endif
}
