/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/audiocapturebase.h"

#include "twitchsdk/core/systemclock.h"

// TODO - See if we can get rid of doing floating point calculations
uint64_t ttv::broadcast::AudioCaptureBase::SamplesToSystemTime(uint64_t samples, uint32_t sampleRate) {
  double samplesInSec = static_cast<double>(samples) / sampleRate;
  return static_cast<uint64_t>(samplesInSec * GetSystemClockFrequency());
}

uint64_t ttv::broadcast::AudioCaptureBase::SystemTimeToSamples(uint64_t time, uint32_t sampleRate) {
  const double samples = static_cast<double>(time) / static_cast<double>(GetSystemClockFrequency());
  return static_cast<uint64_t>(samples * static_cast<double>(sampleRate));
}

ttv::broadcast::AudioCaptureBase::AudioCaptureBase()
    : mAudioLayer(0), mInitialSysTime(0), mMuted(false), mInitialized(false), mStarted(false) {}

void ttv::broadcast::AudioCaptureBase::SetAudioMixer(const std::shared_ptr<IAudioMixer>& mixer) {
  mAudioMixer = mixer;
}

bool ttv::broadcast::AudioCaptureBase::GetMuted() const {
  return mMuted;
}

void ttv::broadcast::AudioCaptureBase::SetMuted(bool muted) {
  mMuted = muted;
}

void ttv::broadcast::AudioCaptureBase::SetAudioLayer(AudioLayerId layer) {
  mAudioLayer = layer;
}

ttv::broadcast::AudioLayerId ttv::broadcast::AudioCaptureBase::GetAudioLayer() const {
  return mAudioLayer;
}

void ttv::broadcast::AudioCaptureBase::SetInitialTime(uint64_t initialTime) {
  mInitialSysTime = initialTime;
}

TTV_ErrorCode ttv::broadcast::AudioCaptureBase::Initialize() {
  if (mInitialized) {
    return TTV_EC_INVALID_STATE;
  }

  mInitialized = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AudioCaptureBase::Start() {
  if (mStarted || !mInitialized || mAudioMixer == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  mStarted = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AudioCaptureBase::Stop() {
  mStarted = false;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AudioCaptureBase::Shutdown() {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (mInitialized) {
    mAudioMixer.reset();

    ec = Stop();

    mInitialized = false;
  }

  return ec;
}
