/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/audiostreamer.h"

#include "twitchsdk/broadcast/audiocapturebase.h"
#include "twitchsdk/broadcast/audioconstants.h"
#include "twitchsdk/broadcast/iaudiocapture.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/internal/passthroughaudiodevice.h"
#include "twitchsdk/broadcast/internal/pcmaudioframe.h"
#include "twitchsdk/broadcast/internal/pcmaudiomixer.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

namespace {
using namespace ttv::broadcast;

const uint32_t kAudioStreamIndex = 1;  // We only support one audio stream in a broadcast.
const bool kPcmInterleaved = true;     // We only handle interleaved PCM audio data currently.
}  // namespace

class ttv::broadcast::AudioStreamer::CaptureContext {
 public:
  struct FrameEntry {
    AudioLayerId layer;
    std::shared_ptr<AudioFrame> frame;
  };

 public:
  CaptureContext() : usePcmMixer(false) {}

 public:
  std::shared_ptr<IAudioFrameReceiver> activeReceiver;
  std::shared_ptr<IAudioFrameReceiver> encoderReceiver;
  ttv::ConcurrentQueue<FrameEntry> pendingFrames;
  bool usePcmMixer;
};

ttv::broadcast::AudioStreamer::LayerProperties::LayerProperties() : initialTime(0), volume(1.0f), enabled(true) {}

ttv::broadcast::AudioStreamer::AudioStreamer()
    : mNumPcmSamplesPerChannel(0), mInitialTime(0), mInitializeResult(TTV_EC_SUCCESS) {
  ttv::trace::Message("AudioStreamer", MessageLevel::Info, "AudioStreamer created");

  mDoProcessing = false;

  TTV_ErrorCode ec = CreateThreadSync(mInitializingThreadSync, "AudioStreamer");
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mInitializingThreadSync != nullptr);
}

ttv::broadcast::AudioStreamer::~AudioStreamer() {
  assert(mDeviceProcessingThread == nullptr);
  assert(!mDoProcessing);

  Stop();

  ttv::trace::Message("AudioStreamer", MessageLevel::Info, "AudioStreamer destroyed");
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::SetEncoder(const std::shared_ptr<IAudioEncoder>& encoder) {
  // TODO: Make sure not streaming
  mAudioEncoder = encoder;

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::AudioStreamer::SetCapturer(AudioLayerId layerId, const std::shared_ptr<IAudioCapture>& capturer) {
  ttv::trace::Message("AudioStreamer", MessageLevel::Debug, "AudioStreamer::SetCapturer()");

  // TODO: Ensure not started

  auto iter = mLayers.find(layerId);
  if (iter != mLayers.end()) {
    if (capturer == nullptr) {
      mLayers.erase(iter);
    } else {
      iter->second.capturer = capturer;
    }
  } else {
    if (capturer != nullptr) {
      LayerProperties properties;
      properties.capturer = capturer;
      mLayers[layerId] = properties;
    }
  }

  if (capturer != nullptr) {
    capturer->SetAudioLayer(layerId);
  }
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::Initialize(const AudioParams& audioParams) {
  AutoTracer tracer("AudioStreamer", MessageLevel::Debug, "AudioStreamer::Initialize()");

  if (mAudioEncoder == nullptr) {
    return TTV_EC_BROADCAST_INVALID_ENCODER;
  }

  TTV_ASSERT(audioParams.numInputAudioLayers > 0);

  mAudioParams = audioParams;

  TTV_ErrorCode ec = mAudioEncoder->Start(kAudioStreamIndex, audioParams);

  // Setup the mixer if there's a chance we might handle PCM audio
  if (TTV_SUCCEEDED(ec)) {
    if (mAudioEncoder->SupportsReceiverProtocol(IPcmAudioFrameReceiver::GetReceiverTypeId())) {
      auto receiver = std::static_pointer_cast<IPcmAudioFrameReceiver>(
        mAudioEncoder->GetReceiverImplementation(IPcmAudioFrameReceiver::GetReceiverTypeId()));

      ec = receiver->GetNumPcmSamplesPerFrame(mNumPcmSamplesPerChannel);

      if (TTV_SUCCEEDED(ec)) {
        TTV_ASSERT(nullptr == mAudioMixer);
        mAudioMixer = std::make_shared<PcmAudioMixer>(audioParams.numChannels, mNumPcmSamplesPerChannel);

        auto bufferSize = mNumPcmSamplesPerChannel * audioParams.numChannels;
        mMixedSamplesBuffer.resize(bufferSize);
      }
    }
  }

  return ec;
}

void ttv::broadcast::AudioStreamer::Stop() {
  AutoTracer tracer("AudioStreamer", MessageLevel::Debug, "AudioStreamer::Stop()");

  // Synchronous stop
  StopCapture();

  if (mAudioEncoder != nullptr) {
    (void)mAudioEncoder->Stop();
  }

  mAudioMixer.reset();

  mInitialTime = 0;

  mMixedSamplesBuffer.resize(0);
}

void ttv::broadcast::AudioStreamer::SetInitialTime(uint64_t initialTime) {
  mInitialTime = initialTime;

  for (const auto& kvp : mLayers) {
    if (kvp.second.capturer != nullptr) {
      kvp.second.capturer->SetInitialTime(mInitialTime);
    }
  }
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::StartCapture() {
  AutoTracer tracer("AudioStreamer", MessageLevel::Debug, "AudioStreamer::StartCapture()");

  // Start the thread for processing capture devices
  //
  mDoProcessing = true;
  TTV_ASSERT(nullptr == mDeviceProcessingThread);

  TTV_ErrorCode ec = ttv::CreateThread(std::bind(&ttv::broadcast::AudioStreamer::ProcessCapturers, this),
    "ttv::broadcast::AudioStreamer::ProcessCapturers", mDeviceProcessingThread);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mDeviceProcessingThread != nullptr);

  // Start the thread
  mDeviceProcessingThread->Run();

  // Wait for capturer initialization to finish on the thread before returning
  ec = mInitializingThreadSync->Suspend();
  TTV_ASSERT(TTV_SUCCEEDED(ec));

  if (TTV_FAILED(mInitializeResult)) {
    StopCapture();
  }

  return mInitializeResult;
}

void ttv::broadcast::AudioStreamer::StopCapture() {
  AutoTracer tracer("AudioStreamer", MessageLevel::Debug, "AudioStreamer::StopCapture()");

  mDoProcessing = false;

  // Kill and wait for the capturer processing thread
  if (mDeviceProcessingThread != nullptr && mDeviceProcessingThread->Joinable()) {
    mDeviceProcessingThread->Join();
  }

  mDeviceProcessingThread.reset();
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::WritePcmAudioFrames(
  const std::shared_ptr<CaptureContext>& context, uint64_t latestTimeToProcess) {
  TTV_ASSERT(mAudioMixer != nullptr);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  uint64_t packetTime;
  ec = mAudioMixer->GetNextPacketTime(packetTime);

  while (TTV_SUCCEEDED(ec) && (packetTime + mNumPcmSamplesPerChannel) < latestTimeToProcess) {
    ec = mAudioMixer->GetPacket(&mMixedSamplesBuffer[0], packetTime);

    if (TTV_SUCCEEDED(ec)) {
      auto pcmReceiver = std::static_pointer_cast<IPcmAudioFrameReceiver>(context->encoderReceiver);
      auto buffer = reinterpret_cast<const uint8_t*>(mMixedSamplesBuffer.data());

      std::shared_ptr<AudioFrame> audioFrame;
      ec = pcmReceiver->PackageFrame(buffer, mNumPcmSamplesPerChannel, mAudioParams.numChannels, kPcmInterleaved,
        AudioSampleFormat::TTV_ASF_PCM_S16, packetTime, audioFrame);

      if (TTV_SUCCEEDED(ec)) {
        ec = mAudioEncoder->SubmitFrame(audioFrame);
      }

      if (TTV_SUCCEEDED(ec)) {
        ec = mAudioMixer->GetNextPacketTime(packetTime);
      }
    }
  }

  if (ec == TTV_EC_BROADCAST_NOMOREDATA) {
    ec = TTV_EC_SUCCESS;
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::WritePassThroughAudioFrames(
  const std::shared_ptr<CaptureContext>& context, uint64_t /*latestTimeToProcess*/) {
  CaptureContext::FrameEntry entry;
  while (context->pendingFrames.try_pop(entry)) {
    mAudioEncoder->SubmitFrame(entry.frame);
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::AudioStreamer::StartCapturers(const std::shared_ptr<IAudioMixer>& mixer) {
  mInitializeResult = TTV_EC_SUCCESS;

  for (const auto& kvp : mLayers) {
    const auto& capturer = kvp.second.capturer;
    if (capturer == nullptr) {
      continue;
    }

    capturer->SetAudioMixer(mixer);

    mInitializeResult = capturer->Start();

    if (TTV_FAILED(mInitializeResult)) {
      ttv::trace::Message("AudioStreamer", MessageLevel::Error,
        "Inside AudioStreamer::StartCapturers - Failed to start IAudioCapture implementation");
      break;
    }
  }
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::StopCapturers() {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  for (const auto& kvp : mLayers) {
    const auto& capturer = kvp.second.capturer;
    if (capturer == nullptr) {
      continue;
    }

    ec = capturer->Stop();

    if (TTV_FAILED(ec)) {
      ttv::trace::Message("AudioStreamer", MessageLevel::Error,
        "Inside AudioStreamer::StopCapturers - Failed to stop IAudioCapture implementation");
    }
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::AudioStreamer::SubmitPcmSamplesToMixer(
  AudioLayerId audioLayerId, const int16_t* samples, size_t numSamples, uint64_t sampleTime) {
  auto iter = mLayers.find(audioLayerId);
  if (iter == mLayers.end()) {
    TTV_ASSERT(false);
    return TTV_EC_BROADCAST_INVALID_AUDIO_LAYER;
  }

  auto& layer = iter->second;

  // Handle volume changes
  while (layer.volumeChanges.size() &&
         layer.volumeChanges.front().first <
           (mInitialTime + AudioCaptureBase::SamplesToSystemTime(sampleTime, kAudioEncodeRate))) {
    layer.volume = layer.volumeChanges.front().second;
    layer.volumeChanges.pop();
  }

  // Add to the mixer
  float volume = layer.volume;
  if (layer.capturer->GetMuted()) {
    volume = 0.f;
  }
  return mAudioMixer->SubmitAudioSamples(samples, sampleTime, numSamples, volume);
};

// This method is called on its own thread
void ttv::broadcast::AudioStreamer::ProcessCapturers() {
  AutoTracer tracer("AudioStreamer", MessageLevel::Debug, "AudioStreamer::ProcessDevices()");

  auto context = std::make_shared<CaptureContext>();

  // A helper class which implements IAudioMixer so IAudioCapture implementations can submit audio
  std::shared_ptr<AudioMixerProxy> mixer = std::make_shared<AudioMixerProxy>();

  // Receives the audio frames from the capturers
  mixer->mSubmitFrameFunc = [this, context](AudioLayerId audioLayerId,
                              const std::shared_ptr<AudioFrame>& audioFrame) -> TTV_ErrorCode {
    TTV_ASSERT(mAudioEncoder->SupportsReceiverProtocol(audioFrame->GetReceiverTypeId()));

    // We pass PCM data through the PcmAudioMixer first
    if (audioFrame->GetReceiverTypeId() == IPcmAudioFrameReceiver::GetReceiverTypeId()) {
      auto pcmFrame = std::static_pointer_cast<PcmAudioFrame>(audioFrame);

      // TODO: Add support for other than 16-bit audio samples
      TTV_ASSERT(pcmFrame->GetAudioSampleFormat() == AudioSampleFormat::TTV_ASF_PCM_S16);
      if (pcmFrame->GetAudioSampleFormat() != AudioSampleFormat::TTV_ASF_PCM_S16) {
        return TTV_EC_UNIMPLEMENTED;
      }

      TTV_ASSERT(pcmFrame->GetNumChannels() == mAudioParams.numChannels);
      TTV_ASSERT(pcmFrame->GetInterleaved() == kPcmInterleaved);

      const int16_t* sampleBuffer = reinterpret_cast<const int16_t*>(pcmFrame->GetSampleBuffer().data());

      return SubmitPcmSamplesToMixer(
        audioLayerId, sampleBuffer, pcmFrame->GetNumSamplesPerChannel(), pcmFrame->GetTimeStamp());
    }
    // We'll just be passing this through to the encoder
    else {
      context->pendingFrames.push({audioLayerId, audioFrame});
    }

    return TTV_EC_SUCCESS;
  };

  mixer->mSupportsReceiverProtocolFunc = [this](IAudioFrameReceiver::ReceiverTypeId typeId) -> bool {
    // We ultimately pass this on to the audio encoder
    return mAudioEncoder->SupportsReceiverProtocol(typeId);
  };

  mixer->mGetReceiverImplementationFunc =
    [this, context](IAudioFrameReceiver::ReceiverTypeId typeId) -> std::shared_ptr<IAudioFrameReceiver> {
    TTV_ASSERT(mAudioEncoder->SupportsReceiverProtocol(typeId));

    // TODO: We can't mix different types of audio yet
    if (context->activeReceiver == nullptr) {
      context->encoderReceiver = mAudioEncoder->GetReceiverImplementation(typeId);

      // We pass PCM data through the PcmAudioMixer first
      if (typeId == IPcmAudioFrameReceiver::GetReceiverTypeId()) {
        context->usePcmMixer = true;
        context->activeReceiver =
          std::make_shared<PcmAudioReceiver>(mNumPcmSamplesPerChannel * mAudioParams.numChannels);
      }
      // We'll just be passing this through to the encoder
      else {
        context->activeReceiver = context->encoderReceiver;
      }
    }

    return context->activeReceiver;
  };

  // Start capturers
  StartCapturers(mixer);

  // Now that the capturers have been started let the other thread continue
  TTV_ErrorCode ec = mInitializingThreadSync->Signal();
  ASSERT_ON_ERROR(ec);

  if (TTV_FAILED(mInitializeResult)) {
    mDoProcessing = false;
  }

  // loop until the streamer is stopped
  while (mDoProcessing) {
    uint64_t earliestSampleTime = 0;

    // Loop over all devices
    bool first = true;
    for (const auto& kvp : mLayers) {
      if (kvp.second.capturer == nullptr) {
        continue;
      }

      // This will cause calls to the IAudioMixer implementation provided above
      uint64_t lastSampleTime = 0;
      ec = kvp.second.capturer->Process(mixer, lastSampleTime);

      // TODO: We don't report errors that happen in the audio system
      ASSERT_ON_ERROR(ec);

      if (first) {
        earliestSampleTime = lastSampleTime;
      } else {
        earliestSampleTime = std::min(earliestSampleTime, lastSampleTime);
      }

      first = false;
    }

    // Grab frames from the mixer
    if (context->usePcmMixer) {
      ec = WritePcmAudioFrames(context, earliestSampleTime);
    }
    // Pass through to the encoder
    else {
      ec = WritePassThroughAudioFrames(context, earliestSampleTime);
    }

    // TODO: We don't report errors that happen in the audio system
    ASSERT_ON_ERROR(ec);

    // TODO:
    // We need to sleep here, just not sure how much. Also we probably want to time
    // how long the above took and scale the sleep accordingly.
    ttv::Sleep(5);
  }

  // Stop capturing
  StopCapturers();
}

float ttv::broadcast::AudioStreamer::GetVolume(AudioLayerId layer) const {
  auto iter = mLayers.find(layer);

  if (iter != mLayers.end()) {
    return iter->second.volume;
  } else {
    return 0.0f;
  }
}

void ttv::broadcast::AudioStreamer::SetVolume(AudioLayerId layer, float volume) {
  TTV_ASSERT(volume >= -1.f && volume <= 1.f);

  auto iter = mLayers.find(layer);
  if (iter != mLayers.end()) {
    iter->second.volumeChanges.push(volumeChanges_t::value_type(GetSystemClockTime(), volume));
  }
}

void ttv::broadcast::AudioStreamer::SetCapturerEnabled(AudioLayerId layer, bool enabled) {
  auto iter = mLayers.find(layer);
  if (iter != mLayers.end()) {
    iter->second.enabled = enabled;
  }
}

void ttv::broadcast::AudioStreamer::GetCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const {
  result.clear();

  for (auto kvp : mLayers) {
    result.push_back(kvp.second.capturer);
  }
}

std::shared_ptr<ttv::broadcast::IAudioCapture> ttv::broadcast::AudioStreamer::GetCapturer(AudioLayerId layer) const {
  auto iter = mLayers.find(layer);
  if (iter != mLayers.end()) {
    return iter->second.capturer;
  }

  return nullptr;
}

void ttv::broadcast::AudioStreamer::GetEnabledCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const {
  result.clear();

  for (auto kvp : mLayers) {
    const auto& data = kvp.second;
    if (data.enabled) {
      result.push_back(data.capturer);
    }
  }
}

bool ttv::broadcast::AudioStreamer::HasEnabledCapturers() const {
  for (auto kvp : mLayers) {
    const auto& data = kvp.second;
    if (data.enabled) {
      return true;
    }
  }

  return false;
}
