/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/broadcastapi.h"

#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/iaudiocapture.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/imuxer.h"
#include "twitchsdk/broadcast/internal/ingesttester.h"
#include "twitchsdk/broadcast/internal/streamer.h"
#include "twitchsdk/broadcast/internal/streamstats.h"
#include "twitchsdk/broadcast/internal/task/runcommercialtask.h"
#include "twitchsdk/broadcast/internal/twitchapi.h"
#include "twitchsdk/broadcast/ivideocapture.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"
#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/trackingcontext.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

#include <fstream>
#include <iostream>
#include <memory>
#include <utility>

namespace {
static const char* kModuleName = "ttv::broadcast::BroadcastAPI";
const char* kRequiredScopes[] = {
  "user_read", "channel_read", "channel_editor", "channel_commercial", "sdk_broadcast", "metadata_events_edit"};
}  // namespace

namespace ttv {
namespace broadcast {
class BroadcastContext {
 public:
  BroadcastContext()
      : mFlvMuxerAsyncEnabled(false),
        mSelectedIngestServer(),
        mLastVideoFrameSubmissionError(TTV_EC_SUCCESS),
        mCurrentState(BroadcastState::Initialized),
        mLastReportedState(BroadcastState::Initialized) {}

  void SetVideoFrameReceiver(std::shared_ptr<IVideoFrameReceiver> receiver) { mVideoFrameReceiver = receiver; }
  std::shared_ptr<IVideoFrameReceiver> GetVideoFrameReceiver() const { return mVideoFrameReceiver; }

  void SetStreamer(std::shared_ptr<Streamer> streamer) { mStreamer = streamer; }
  std::shared_ptr<Streamer> GetStreamer() const { return mStreamer.lock(); }

  void SetVideoEncoder(std::shared_ptr<IVideoEncoder> encoder) { mVideoEncoder = encoder; }
  std::shared_ptr<IVideoEncoder> GetVideoEncoder() const { return mVideoEncoder; }

  void SetAudioEncoder(std::shared_ptr<IAudioEncoder> encoder) { mAudioEncoder = encoder; }
  std::shared_ptr<IAudioEncoder> GetAudioEncoder() const { return mAudioEncoder; }

  void SetVideoParams(const VideoParams& params) {
    mVideoParams = params;

    // Clamp values
    mVideoParams.maximumKbps = std::min(mVideoParams.maximumKbps, kMaxBitRate);
    mVideoParams.maximumKbps = std::max(mVideoParams.maximumKbps, kMinBitRate);
    mVideoParams.minimumKbps = std::min(mVideoParams.minimumKbps, kMaxBitRate);
    mVideoParams.minimumKbps = std::max(mVideoParams.minimumKbps, kMinBitRate);
    mVideoParams.initialKbps = std::min(mVideoParams.initialKbps, mVideoParams.maximumKbps);
    mVideoParams.initialKbps = std::max(mVideoParams.initialKbps, mVideoParams.minimumKbps);
  }
  VideoParams& GetVideoParams() { return mVideoParams; }

  void SetSelectedIngestServer(const IngestServer& server) { mSelectedIngestServer = server; }
  IngestServer& GetSelectedIngestServer() { return mSelectedIngestServer; }

  void SetLastVideoFrameSubmissionError(TTV_ErrorCode ec) { mLastVideoFrameSubmissionError = ec; }
  TTV_ErrorCode GetLastVideoFrameSubmissionError() const { return mLastVideoFrameSubmissionError; }

  void SetCurrentState(BroadcastState state) { mCurrentState = state; }
  BroadcastState GetCurrentState() const { return mCurrentState; }

  void SetLastReportedState(BroadcastState state) { mLastReportedState = state; }
  BroadcastState GetLastReportedState() const { return mLastReportedState; }

  void SetFlvFileName(const std::wstring& path) { mFlvFilename = path; }
  std::wstring GetFlvFileName() const { return mFlvFilename; }

  void SetFlvMuxerAsyncEnabled(bool enable) { mFlvMuxerAsyncEnabled = enable; }
  bool GetFlvMuxerAsyncEnabled() { return mFlvMuxerAsyncEnabled; }

 private:
  std::shared_ptr<IVideoFrameReceiver> mVideoFrameReceiver;

  std::shared_ptr<IVideoEncoder> mVideoEncoder;
  std::shared_ptr<IAudioEncoder> mAudioEncoder;

  bool mFlvMuxerAsyncEnabled;
  VideoParams mVideoParams;
  StreamInfo mStreamInfo;
  IngestServer mSelectedIngestServer;
  // std::shared_ptr<ArchivingState> mArchiveState;

  TTV_ErrorCode mLastVideoFrameSubmissionError;
  BroadcastState mCurrentState;
  BroadcastState mLastReportedState;

  std::weak_ptr<Streamer> mStreamer;

  std::wstring mFlvFilename;
};

struct IngestTesterEntry {
  std::shared_ptr<IngestTester> ingestTester;
  std::weak_ptr<User> user;
};

/**
 * Module-level internal data.
 */
class BroadcastApiInternalData {
 public:
  BroadcastApiInternalData()
      : broadcasterSoftware("sdk"),
        streamerContext(std::make_shared<StreamerContext>()),
        activeUserId(0),
        forceArchiveBroadcast(false),
        forceDontArchiveBroadcast(false) {
    TTV_ErrorCode ret = ttv::CreateMutex(mutex, "BroadcastApiInternalData");
    ASSERT_ON_ERROR(ret);
  }

  std::shared_ptr<CoreAPI> coreApi;
  std::shared_ptr<UserRepository> userRepository;  //!> The set of all local users.
  std::shared_ptr<SettingRepository> settingRepository;
  std::shared_ptr<TaskRunner> taskRunner;               //!> The general task runner to use for api calls.
  std::shared_ptr<IEventScheduler> mainEventScheduler;  //!> The main event scheduler to run tasks on.
  std::shared_ptr<TwitchAPI> twitchApi;                 //!> The global api requester which is independent of Users.
  std::shared_ptr<StreamerListenerProxy> streamerListener;
  std::shared_ptr<BroadcastContext> broadcastContext;
  std::shared_ptr<IMuxer> customMuxer;
  std::vector<IngestServer> ingestServerList;
  std::string broadcasterSoftware;
  std::shared_ptr<User> activeUser;  //!< The user which all broadcast commands apply.

  std::unique_ptr<IMutex> mutex;  // The mutex that guards the instance containers here
  std::vector<IngestTesterEntry> ingestTesterInstances;

  // Shared state between ingest testers and the BroadcastAPI to ensure we
  // are never ingest testing and broadcasting at the same time.
  std::shared_ptr<StreamerContext> streamerContext;

  UserId activeUserId;  //!< The userid of the user which all broadcast commands apply.

  bool forceArchiveBroadcast;
  bool forceDontArchiveBroadcast;

  bool IsBroadcastInProgress() const {
    auto state = broadcastContext->GetCurrentState();
    return state == BroadcastState::StartingBroadcast || state == BroadcastState::Broadcasting ||
           state == BroadcastState::StoppingBroadcast;
  }
};
}  // namespace broadcast
}  // namespace ttv

ttv::broadcast::BroadcastAPI::BroadcastAPI() {
  // Register the error lookup function for the module
  ttv::RegisterErrorToStringFunction(&ttv::broadcast::BroadcastErrorToString);
  ttv::RegisterErrorCodeValueFunction(&ttv::broadcast::GetBroadcastErrorCodeValues);

  mInternalData = std::make_shared<BroadcastApiInternalData>();
}

std::string ttv::broadcast::BroadcastAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  TTV_ASSERT(coreApi != nullptr);

  mInternalData->coreApi = coreApi;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetListener(const std::shared_ptr<IBroadcastAPIListener>& listener) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetCustomMuxer(const std::shared_ptr<IMuxer>& muxer) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure not broadcasting
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->customMuxer = muxer;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetBroadcasterSoftware(const std::string& str) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mInternalData->broadcasterSoftware = str;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetForceArchiveBroadcast(bool forceArchive) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mInternalData->forceArchiveBroadcast = forceArchive;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetForceDontArchiveBroadcast(bool forceDontArchive) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mInternalData->forceDontArchiveBroadcast = forceDontArchive;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::CreateIngestTester(UserId userId,
  const std::shared_ptr<IIngestTesterListener>& listener, const uint8_t* testDataBuffer, uint32_t testDataLength,
  std::shared_ptr<IIngestTester>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<User> user = mInternalData->userRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  std::shared_ptr<IngestTester> ingestTester = std::make_shared<IngestTester>(user, mInternalData->streamerContext);
  ingestTester->SetDisposer(
    [ingestTester, data = mInternalData]() mutable { DisposeIngestTesterInternal(ingestTester, data); });

  TTV_ErrorCode ec = ingestTester->SetTestData(testDataBuffer, testDataLength);
  if (TTV_SUCCEEDED(ec)) {
    ingestTester->SetMainEventScheduler(mInternalData->mainEventScheduler);
    ingestTester->SetTaskRunner(mInternalData->taskRunner);
    ingestTester->AddListener(listener);
    ec = ingestTester->Initialize();
  }

  // Register the component
  if (TTV_SUCCEEDED(ec)) {
    IngestTesterEntry entry;
    entry.ingestTester = ingestTester;
    entry.user = user;

    {
      AutoMutex lock(mInternalData->mutex.get());
      mInternalData->ingestTesterInstances.push_back(entry);
    }

    user->GetComponentContainer()->AddComponent(ingestTester);

    // Create and return a public reference to this
    result.reset(ingestTester.get(), [ingestTester](const IngestTester* /*p*/) { ingestTester->Dispose(); });
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::DisposeIngestTester(const std::shared_ptr<IIngestTester>& ingestTester) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  return BroadcastAPI::DisposeIngestTesterInternal(ingestTester, mInternalData);
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::DisposeIngestTesterInternal(
  const std::shared_ptr<IIngestTester>& ingestTester, const std::shared_ptr<BroadcastApiInternalData>& internalData) {
  TTV_RETURN_ON_NULL(ingestTester, TTV_EC_INVALID_ARG);

  TTV_RETURN_ON_NULL(internalData, TTV_EC_NOT_INITIALIZED);

  AutoMutex lock(internalData->mutex.get());

  // Make sure we know about this object
  auto iter = std::find_if(internalData->ingestTesterInstances.begin(), internalData->ingestTesterInstances.end(),
    [&ingestTester](const IngestTesterEntry& x) { return x.ingestTester == ingestTester; });

  if (iter == internalData->ingestTesterInstances.end()) {
    return TTV_EC_INVALID_ARG;
  }

  // Get the concrete object
  auto obj = std::static_pointer_cast<IngestTester>(ingestTester);

  std::shared_ptr<User> user = obj->GetUser();
  if (user != nullptr) {
    auto container = user->GetComponentContainer();
    if (container != nullptr) {
      TTV_ASSERT(container->ContainsComponent(obj));

      container->DisposeComponent(obj);
    }
  }

  // Dispose of it
  internalData->ingestTesterInstances.erase(iter);

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::BroadcastAPI::GetRequiredAuthScopes(std::vector<std::string>& scopes) {
  Streamer::GetRequiredAuthScopes(scopes);
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::BindToUser(const std::shared_ptr<User>& user) {
  std::string userName;

  if (user != nullptr) {
    userName = user->GetUserName();
  }

  // Clear the previous binding
  mInternalData->broadcastContext->SetStreamer(nullptr);
  mInternalData->activeUser.reset();

  // Bind to the new user
  if (user != nullptr) {
    mInternalData->activeUser = user;

    // We assume the streamer exists
    auto streamer = mInternalData->activeUser->GetComponentContainer()->GetComponent<Streamer>();
    TTV_ASSERT(streamer != nullptr);

    mInternalData->broadcastContext->SetStreamer(streamer);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetActiveUser(UserId userId) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Can't change the user binding while broadcasting
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }
  // No change in user
  else if (userId == mInternalData->activeUserId) {
    return TTV_EC_SUCCESS;
  }

  mInternalData->activeUserId = userId;

  // Grab the user, binding if the user is logged in
  auto user = mInternalData->userRepository->GetUser(userId);
  BindToUser(user);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::Initialize(const InitializeCallback& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "Initialize");

  TTV_RETURN_ON_NULL(mInternalData->coreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;
    mCoreApiClient = std::make_shared<CoreApiClient>(this);

    mInternalData->mainEventScheduler = mInternalData->coreApi->GetMainEventScheduler();
    mInternalData->taskRunner = std::make_shared<TaskRunner>("BroadcastAPI");
    mInternalData->twitchApi = std::make_shared<TwitchAPI>();
    mInternalData->userRepository = mInternalData->coreApi->GetUserRepository();
    mInternalData->settingRepository = mInternalData->coreApi->GetSettingRepository();
    mInternalData->broadcastContext = std::make_shared<BroadcastContext>();

    const auto& rootTrackingContext = mInternalData->coreApi->GetTrackingContext();
    TTV_ASSERT(rootTrackingContext != nullptr);

    mInternalData->streamerContext->sharedTrackingContext = std::make_shared<TrackingContext>(rootTrackingContext);

    mInternalData->twitchApi->SetTaskRunner(mInternalData->taskRunner);
    mInternalData->twitchApi->Initialize();
    GetComponentContainer()->AddComponent(mInternalData->twitchApi);

    ec = mInternalData->coreApi->RegisterClient(mCoreApiClient);
  }

  if (TTV_SUCCEEDED(ec)) {
    NotifyStateChange();
    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::Shutdown(const ShutdownCallback& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "Shutdown");

  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);

  if (TTV_SUCCEEDED(ec)) {
    // Cleanup data in users
    std::vector<std::shared_ptr<User>> users;
    mInternalData->userRepository->GetUsers(users);

    for (auto user : users) {
      CoreUserLoggedOut(user);
    }

    RegisterShutdownCallback(callback);
  }

  return ec;
}

bool ttv::broadcast::BroadcastAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (!mInternalData->taskRunner->IsShutdown()) {
    mInternalData->taskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::broadcast::BroadcastAPI::CompleteShutdown() {
  if (mInternalData->taskRunner != nullptr) {
    mInternalData->taskRunner->CompleteShutdown();
    mInternalData->taskRunner.reset();
  }

  if (mCoreApiClient != nullptr) {
    if (mInternalData->coreApi != nullptr) {
      mInternalData->coreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  mInternalData->coreApi.reset();
  mInternalData->twitchApi.reset();
  mInternalData->settingRepository.reset();
  mInternalData->broadcastContext.reset();
  mInternalData->userRepository.reset();
  mInternalData->activeUser.reset();
  mInternalData->activeUserId = 0;
  mInternalData->streamerListener.reset();
  mInternalData->ingestServerList.clear();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetBroadcastState(BroadcastState& result) const {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  result = mInternalData->broadcastContext->GetCurrentState();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetStreamInfo(UserId userId, ChannelId channelId, const std::string& game,
  const std::string& title, SetStreamInfoCallback&& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "UpdateStreamInfo");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  auto user = mInternalData->userRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto api = user->GetComponentContainer()->GetComponent<TwitchAPI>();
  if (api == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto taskCallback = [callback = std::move(callback)](TwitchAPI* /*source*/, TTV_ErrorCode ec) {
    if (callback != nullptr) {
      callback(ec);
    }
  };

  if (userId == mInternalData->activeUserId) {
    mInternalData->streamerContext->category = game;
    mInternalData->streamerContext->title = title;
  }

  return api->SetStreamInfo(channelId, title, game, std::move(taskCallback));
}

// TTV_ErrorCode ttv::broadcast::BroadcastAPI::FetchRecordingStatus(const std::string& userName)
//{
//  ttv::AutoTracer EntryAndExitTrace( "BroadcastAPI", MessageLevel::Debug, "RequestRecordingStatus" );
//
//  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
//  TTV_RETURN_ON_DIFFERENT(IsValidUserName(userName), true, TTV_EC_INVALID_LOGIN);
//
//  auto user = mInternalData->userRepository->GetUser(userName);
//  if (user == nullptr)
//  {
//      return TTV_EC_NEED_TO_LOGIN;
//  }
//
//  auto api = user->GetComponentContainer()->GetComponent<TwitchAPI>();
//  if (api == nullptr)
//  {
//      return TTV_EC_NEED_TO_LOGIN;
//  }
//
//  TwitchAPI::GetChannelRecordingStatusCallback callback = [this](TwitchAPI* /*source*/, TTV_ErrorCode ec, const
//  std::shared_ptr<ArchivingState>& result)
//  {
//      // TODO: Finish this
//      //Invoke<IBroadcastAPIListener>([this, ec, &result](std::shared_ptr<IBroadcastAPIListener> listener)
//      //{
//      //  listener->BroadcastRecordingStatusFetchComplete(ec, result);
//      //});
//  };
//
//  return api->GetChannelRecordingStatus(std::move(callback));
//}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::RunCommercial(
  UserId userId, ChannelId channelId, uint32_t timebreakSeconds, RunCommercialCallback&& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "RunCommercial");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  auto user = mInternalData->userRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto api = user->GetComponentContainer()->GetComponent<TwitchAPI>();
  if (api == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  TwitchAPI::RunCommercialCallback taskCallback = [callback = std::move(callback)](
                                                    TwitchAPI* /*source*/, TTV_ErrorCode ec) {
    if (callback != nullptr) {
      callback(ec);
    }
  };

  return api->RunCommercial(channelId, timebreakSeconds, std::move(taskCallback));
}

// TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetDefaultPixelFormat(TTV_PixelFormat& result) const
//{
//  result = TTV_PF_RGBA;
//
//  // TODO: This code belongs in the IFrameSubmitter interface since it's platform-specific
//#ifdef TTV_PLATFORM_WIN32
//  result = TTV_PF_BGRA;
//#endif
//
//#ifdef TTV_PLATFORM_IOS
//  result = TTV_PF_BGRA;
//#endif
//
//#ifdef TTV_PLATFORM_MAC
//  result = TTV_PF_RGBA;
//#endif
//
//  return TTV_EC_SUCCESS;
//}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetVideoParams(const VideoParams& videoParams) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "SetVideoParams");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure not broadcasting
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->broadcastContext->SetVideoParams(videoParams);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetVideoParams(VideoParams& videoParams) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  videoParams = mInternalData->broadcastContext->GetVideoParams();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetOutputPath(const std::wstring& outputPath) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "SetOutputPath");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure broadcast not in progress
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->broadcastContext->SetFlvFileName(outputPath);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetConnectionType(ConnectionType connectionType) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "SetConnectionType");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure broadcast not in progress
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->streamerContext->connectionType = connectionType;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetSessionId(const std::string& sessionId) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "SetSessionId");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure broadcast not in progress
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->streamerContext->sharedTrackingContext->SetProperty("debug_session_id", sessionId);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::AddBandwidthStatListener(
  const std::shared_ptr<IBandwidthStatListener>& listener) {
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);
  if (TTV_SUCCEEDED(ec)) {
    streamer->AddBandwidthStatListener(listener);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::RemoveBandwidthStatListener(
  const std::shared_ptr<IBandwidthStatListener>& listener) {
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);
  if (TTV_SUCCEEDED(ec)) {
    streamer->RemoveBandwidthStatListener(listener);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::StartBroadcast(StartBroadcastCallback&& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "StartBroadcast");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // The active user has not been set or logged in
  if (mInternalData->activeUser == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  IngestServer ingestServer;
  VideoParams videoParams;
  const auto& context = mInternalData->broadcastContext;

  if (TTV_SUCCEEDED(ec)) {
    ingestServer = context->GetSelectedIngestServer();

    if (mInternalData->customMuxer == nullptr) {
      // Select the default server if none set
      if (ingestServer.serverUrl.empty() && context->GetFlvFileName().empty()) {
        if (!mInternalData->ingestServerList.empty()) {
          ingestServer = mInternalData->ingestServerList[0];
          mInternalData->broadcastContext->SetSelectedIngestServer(ingestServer);
        } else {
          ttv::trace::Message(
            "BroadcastAPI", MessageLevel::Error, "BroadcastAPI::StartBroadcast - No ingest server specified");

          return TTV_EC_BROADCAST_INVALID_INGEST_SERVER;
        }
      }
    }

    // Make sure we have all the data we need to start it
    if (mInternalData->broadcastContext->GetCurrentState() < BroadcastState::ReadyToBroadcast) {
      return TTV_EC_REQUEST_PENDING;
    }

    videoParams = context->GetVideoParams();

    // Validate video params
    ec = streamer->ValidateParams(videoParams, ingestServer);
  }

  if (TTV_SUCCEEDED(ec)) {
    // Kick off an asynchronous start
    Streamer::StartParams params;
    params.videoParams = videoParams;
    params.ingestServer = ingestServer;
    params.flags = StreamStartFlags::None;
    params.enableAsyncFlvMuxer = context->GetFlvMuxerAsyncEnabled();
    params.outputFile = context->GetFlvFileName();

    streamer->SetCustomMuxer(mInternalData->customMuxer);
    streamer->SetBroadcasterSoftware(mInternalData->broadcasterSoftware);
    streamer->SetForceArchiveBroadcast(mInternalData->forceArchiveBroadcast);
    streamer->SetForceDontArchiveBroadcast(mInternalData->forceDontArchiveBroadcast);

    ec = streamer->Start(params, std::move(callback));
  }

  if (TTV_SUCCEEDED(ec)) {
    SetBroadcastState(ec, BroadcastState::StartingBroadcast);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetStreamer(std::shared_ptr<Streamer>& result) {
  result.reset();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  result = mInternalData->broadcastContext->GetStreamer();
  if (result == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::EnsureBroadcastingAndGetStreamer(std::shared_ptr<Streamer>& result) {
  TTV_ErrorCode ec = GetStreamer(result);
  if (TTV_SUCCEEDED(ec)) {
    // Make sure broadcast in progress
    if (!mInternalData->IsBroadcastInProgress()) {
      return TTV_EC_BROADCAST_STREAM_NOT_STARTED;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::EnsureNotBroadcastingAndGetStreamer(std::shared_ptr<Streamer>& result) {
  TTV_ErrorCode ec = GetStreamer(result);
  if (TTV_SUCCEEDED(ec)) {
    // Make sure not broadcasting
    if (mInternalData->IsBroadcastInProgress()) {
      return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
    }
  }
  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::StopBroadcast(const std::string& reason, StopBroadcastCallback&& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "StopBroadcast");

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    auto context = mInternalData->broadcastContext;

    // Stop video submission
    auto video = streamer->GetVideoCapturer();
    if (video != nullptr) {
      video->Stop();
    }

    // Kick off an asynchronous stop
    ec = streamer->Stop(reason, std::move(callback));
  }

  if (TTV_SUCCEEDED(ec)) {
    SetBroadcastState(ec, BroadcastState::StoppingBroadcast);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::FetchIngestServerList(FetchIngestListCallback&& callback) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "FetchIngestServerList");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // The active user has not been set or logged in
  if (mInternalData->activeUser == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  TwitchAPI::GetIngestServerListCallback taskCallback = [this, callback = std::move(callback)](TwitchAPI* /*source*/,
                                                          TTV_ErrorCode ec, std::vector<IngestServer>&& result) {
    if (TTV_SUCCEEDED(ec)) {
      mInternalData->ingestServerList = result;
    }

    // Notify the client
    if (callback != nullptr) {
      callback(ec, std::move(result));
    }
  };

  auto api = mInternalData->activeUser->GetComponentContainer()->GetComponent<TwitchAPI>();
  if (api == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  return api->GetIngestServerList(std::move(taskCallback));
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetCurrentBroadcastTime(uint64_t& broadcastTime) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  if (!mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_NOT_STARTED;
  }

  auto streamer = mInternalData->broadcastContext->GetStreamer();
  TTV_ASSERT(streamer != nullptr);

  broadcastTime = streamer->GetStreamTime();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetSelectedIngestServer(IngestServer& result) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  result = mInternalData->broadcastContext->GetSelectedIngestServer();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetSelectedIngestServer(const IngestServer& server) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // TODO: We might want to validate here
  mInternalData->broadcastContext->SetSelectedIngestServer(server);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    ec = streamer->SetVideoEncoder(encoder);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetAudioEncoder(const std::shared_ptr<IAudioEncoder>& encoder) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    ec = streamer->SetAudioEncoder(encoder);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetVideoCapturer(const std::shared_ptr<IVideoCapture>& capturer) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    streamer->SetVideoCapturer(capturer);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetAudioCapturer(
  AudioLayerId layer, const std::shared_ptr<IAudioCapture>& capturer) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    streamer->SetAudioCapturer(layer, capturer);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::RemoveAudioCapturer(AudioLayerId layer) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    streamer->SetAudioCapturer(layer, nullptr);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetAudioLayerVolume(AudioLayerId layer, float volume) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = GetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    streamer->SetVolume(layer, volume);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetAudioLayerMuted(AudioLayerId layer, bool muted) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = GetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    auto capturer = streamer->GetAudioCapturer(layer);
    if (capturer != nullptr) {
      capturer->SetMuted(muted);
    } else {
      ec = TTV_EC_BROADCAST_INVALID_AUDIO_LAYER;
    }
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetAudioLayerEnabled(AudioLayerId layer, bool enabled) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  std::shared_ptr<Streamer> streamer;
  TTV_ErrorCode ec = EnsureNotBroadcastingAndGetStreamer(streamer);

  if (TTV_SUCCEEDED(ec)) {
    streamer->SetAudioCapturerEnabled(layer, enabled);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::Update() {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "Update");

  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mInternalData->taskRunner != nullptr) {
    mInternalData->taskRunner->PollTasks();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;

      SetBroadcastState(TTV_EC_SUCCESS, BroadcastState::ReadyToBroadcast);

      NotifyStateChange();

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::BroadcastAPI::SetBroadcastState(TTV_ErrorCode ec, BroadcastState state) {
  auto context = mInternalData->broadcastContext;

  BroadcastState previousState = context->GetCurrentState();

  // Prevent redundant state changes
  if (state == previousState) {
    return;
  }

  ttv::trace::Message(
    "BroadcastAPI", MessageLevel::Debug, "BroadcastAPI::SetBroadcastState - Setting next state to %d", state);

  // Update state
  context->SetCurrentState(state);

  // Notify the listener
  Invoke<IBroadcastAPIListener>(
    [ec, state](std::shared_ptr<IBroadcastAPIListener> listener) { listener->BroadcastStateChanged(ec, state); });
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::SetFlvMuxerAsyncEnabled(bool enable) {
  ttv::AutoTracer EntryAndExitTrace("BroadcastAPI", MessageLevel::Debug, "SetFlvMuxerAsyncEnabled");

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  // Make sure not broadcasting
  if (mInternalData->IsBroadcastInProgress()) {
    return TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
  }

  mInternalData->broadcastContext->SetFlvMuxerAsyncEnabled(enable);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::BroadcastAPI::GetFlvMuxerAsyncEnabled(bool& isEnabledResult) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  isEnabledResult = mInternalData->broadcastContext->GetFlvMuxerAsyncEnabled();
  return TTV_EC_SUCCESS;
}

std::shared_ptr<ttv::broadcast::Streamer> ttv::broadcast::BroadcastAPI::CreateStreamer(
  const std::shared_ptr<User>& user) {
  std::shared_ptr<Streamer> streamer = std::make_shared<Streamer>(user, mInternalData->streamerContext);
  streamer->AddListener(mInternalData->streamerListener);
  streamer->Initialize();

  user->GetComponentContainer()->SetComponent(Streamer::GetComponentName(), streamer);

  return streamer;
}

std::shared_ptr<ttv::broadcast::TwitchAPI> ttv::broadcast::BroadcastAPI::CreateTwitchAPI(
  const std::shared_ptr<User>& user) {
  std::shared_ptr<TwitchAPI> api = std::make_shared<TwitchAPI>(user);
  api->SetTaskRunner(mInternalData->taskRunner);
  api->Initialize();

  user->GetComponentContainer()->SetComponent(TwitchAPI::GetComponentName(), api);

  return api;
}

ttv::broadcast::BroadcastAPI::CoreApiClient::CoreApiClient(BroadcastAPI* api) : mApi(api) {}

std::string ttv::broadcast::BroadcastAPI::CoreApiClient::GetClientName() {
  return kModuleName;
}

void ttv::broadcast::BroadcastAPI::CoreApiClient::CoreUserLoggedIn(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedIn(user);
}

void ttv::broadcast::BroadcastAPI::CoreApiClient::CoreUserLoggedOut(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedOut(user);
}

void ttv::broadcast::BroadcastAPI::CoreApiClient::GetRequiredOAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

void ttv::broadcast::BroadcastAPI::CoreUserLoggedIn(std::shared_ptr<User> user) {
  if (mState == State::Uninitialized || mState == State::ShuttingDown) {
    return;
  }

  if (mInternalData->streamerListener == nullptr) {
    mInternalData->streamerListener = std::make_shared<StreamerListenerProxy>();

    mInternalData->streamerListener->mStreamerStateChangedFunc =
      [this](Streamer* /*source*/, Streamer::StreamerState streamerState, TTV_ErrorCode ec) {
        // Notify clients of state change

        BroadcastState state;

        switch (streamerState) {
          case Streamer::StreamerState::Starting:
            state = BroadcastState::StartingBroadcast;
            break;
          case Streamer::StreamerState::Started:
            state = BroadcastState::Broadcasting;
            break;
          case Streamer::StreamerState::Stopping:
            state = BroadcastState::StoppingBroadcast;
            break;
          default:
          case Streamer::StreamerState::Stopped:
            state = BroadcastState::ReadyToBroadcast;
            break;
        }

        SetBroadcastState(ec, state);
      };
  }

  mInternalData->streamerListener->mBandwidthWarningFunc = [this](Streamer* /*source*/, TTV_ErrorCode ec,
                                                             uint32_t backupMilliseconds) {
    // Notify clients of bandwidth issues
    Invoke<IBroadcastAPIListener>([ec, backupMilliseconds](const std::shared_ptr<IBroadcastAPIListener>& listener) {
      listener->BroadcastBandwidthWarning(ec, backupMilliseconds);
    });
  };

  mInternalData->streamerListener->mStreamInfoFetchedFunc = [this](Streamer* /*source*/, TTV_ErrorCode ec,
                                                              const StreamInfo& streamInfo) {
    // Notify clients of broadcast id
    Invoke<IBroadcastAPIListener>([ec, &streamInfo](const std::shared_ptr<IBroadcastAPIListener>& listener) {
      listener->StreamInfoFetched(ec, streamInfo);
    });
  };

  mInternalData->streamerListener->mStreamKeyErrorFunc = [this](
                                                           Streamer* /*source*/, const CanTheyError& canTheyError) {
    // Notify clients of streamkey error
    Invoke<IBroadcastAPIListener>([&canTheyError](const std::shared_ptr<IBroadcastAPIListener>& listener) {
      listener->StreamKeyError(canTheyError);
    });
  };

  std::shared_ptr<Streamer> streamer = user->GetComponentContainer()->GetComponent<Streamer>();
  if (streamer == nullptr) {
    streamer = CreateStreamer(user);
  }

  std::shared_ptr<TwitchAPI> api = user->GetComponentContainer()->GetComponent<TwitchAPI>();
  if (api == nullptr) {
    api = CreateTwitchAPI(user);
  }

  // Bind to the active user
  if (user->GetUserId() == mInternalData->activeUserId) {
    BindToUser(user);
  }
}

void ttv::broadcast::BroadcastAPI::CoreUserLoggedOut(std::shared_ptr<User> user) {
  // We don't need these components anymore
  auto container = user->GetComponentContainer();
  if (container != nullptr) {
    // NOTE: This will implicitly end the broadcast if in progress
    container->DisposeComponent(Streamer::GetComponentName());
    container->DisposeComponent(TwitchAPI::GetComponentName());
  }

  // Unbind from the user
  if (mInternalData->activeUser == user) {
    mInternalData->activeUser.reset();
  }
}
