/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/broadcasttypes.h"

#include "twitchsdk/broadcast/iaudiocapture.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/iaudiomixer.h"
#include "twitchsdk/broadcast/ivideocapture.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"

namespace {
using namespace ttv::broadcast;

/**
 * Computes the maximum feasible resolution given the expected bandwidth, desired framerate,
 * aspect ratio and number of bits per pixel.
 */
TTV_ErrorCode GetMaxResolution(
  uint32_t expectedKbps, uint32_t frameRate, float bitsPerPixel, float aspectRatio, uint32_t& width, uint32_t& height) {
  // Validate max bitrate.
  if (expectedKbps > kMaxBitRate || expectedKbps < kMinBitRate) {
    return TTV_EC_BROADCAST_INVALID_BITRATE;
  }

  // Validate frames per second.
  if (frameRate < kMinFramesPerSecond || frameRate > kMaxFramesPerSecond) {
    return TTV_EC_BROADCAST_INVALID_FPS;
  }

  // Validate aspect ratio and bits per pixel.
  if (aspectRatio < 0.f || bitsPerPixel < 0.f) {
    return TTV_EC_INVALID_ARG;
  }

  uint32_t totalPixels =
    static_cast<uint32_t>(((expectedKbps * 1000.f) / bitsPerPixel) / static_cast<float>(frameRate));

  // totalPixels = width * height
  // ==> totalPixels = height * height * aspectRatio
  // ==> height = sqrt(totalPixels / aspectRatio)
  height = static_cast<uint32_t>(std::sqrt(static_cast<float>(totalPixels) / aspectRatio));
  height = ((height + 8) / 16) * 16;  // Round to nearest multiple of 16

  width = static_cast<uint32_t>(static_cast<float>(height) * aspectRatio);
  width = ((width + 16) / 32) * 32;  // Round to nearest multiple of 32

  // Clamp the width and height while maintaining the aspect ratio and requirements
  // of multiple of 16 and 32 for height and width
  if (height > kMaxFrameHeight) {
    height = kMaxFrameHeight;
    width = static_cast<uint32_t>(static_cast<float>(height) * aspectRatio);
    width = ((width + 16) / 32) * 32;
  }
  if (width > kMaxFrameWidth) {
    width = kMaxFrameWidth;
    height = static_cast<uint32_t>(static_cast<float>(width) / aspectRatio);
    height = (height / 16) * 16;  // Round down to a multiple of 16 to ensure we don't exceed max height
  }

  return TTV_EC_SUCCESS;
}
}  // namespace

ttv::broadcast::AudioParams::AudioParams()
    : numInputAudioLayers(0), numChannels(0), encodedSampleRate(static_cast<uint32_t>(AudioSampleRate::Hz44100)) {}

ttv::broadcast::IAudioEncoder::IAudioEncoder() {}

ttv::broadcast::IngestServer::IngestServer() : priority(0), serverId(0) {}

ttv::broadcast::IngestServer::IngestServer(const IngestServer& other) {
  serverName = other.serverName;
  serverUrl = other.serverUrl;
  priority = other.priority;
  serverId = other.serverId;
}

void ttv::broadcast::IngestServer::operator=(const IngestServer& other) {
  serverName = other.serverName;
  serverUrl = other.serverUrl;
  priority = other.priority;
  serverId = other.serverId;
}

ttv::broadcast::VideoParams::VideoParams()
    : outputWidth(0),
      outputHeight(0),
      targetFramesPerSecond(kDefaultFramesPerSecond),
      initialKbps(kDefaultInitialBitRate),
      minimumKbps(kMinBitRate),
      maximumKbps(kMaxBitRate),
      encodingCpuUsage(EncodingCpuUsage::Default),
      automaticBitRateAdjustmentEnabled(true) {}

TTV_ErrorCode ttv::broadcast::VideoParams::ConfigureForBandwidth(
  uint32_t expectedKbps, uint32_t framesPerSecond, float bitsPerPixel, float aspectRatio, VideoParams& result) {
  uint32_t width = 0;
  uint32_t height = 0;
  TTV_ErrorCode ec = GetMaxResolution(expectedKbps, framesPerSecond, bitsPerPixel, aspectRatio, width, height);

  if (TTV_SUCCEEDED(ec)) {
    result.initialKbps = expectedKbps;
    if (result.initialKbps > result.maximumKbps) {
      result.initialKbps = result.maximumKbps;
    } else if (result.initialKbps < result.minimumKbps) {
      result.initialKbps = result.minimumKbps;
    }

    result.targetFramesPerSecond = framesPerSecond;
    result.outputWidth = width;
    result.outputHeight = height;
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::VideoParams::ConfigureForResolution(
  uint32_t width, uint32_t height, uint32_t framesPerSecond, float bitsPerPixel, VideoParams& result) {
  if (framesPerSecond < kMinFramesPerSecond || framesPerSecond > kMaxFramesPerSecond) {
    return TTV_EC_BROADCAST_INVALID_FPS;
  } else if (width < 32 || width > kMaxFrameWidth) {
    return TTV_EC_BROADCAST_INVALID_RESOLUTION;
  } else if (height < 16 || height > kMaxFrameHeight) {
    return TTV_EC_BROADCAST_INVALID_RESOLUTION;
  }

  result.targetFramesPerSecond = framesPerSecond;
  result.outputWidth = width;
  result.outputHeight = height;

  // Calculate the bitrate based on resolution
  uint32_t pixelCount = height * width;
  uint32_t pixelsPerSec = pixelCount * framesPerSecond;

  // Set remaining parameters
  result.initialKbps = static_cast<uint32_t>((pixelsPerSec / 1000.f) * bitsPerPixel);
  result.initialKbps = std::min(result.initialKbps, result.maximumKbps);
  result.initialKbps = std::max(result.initialKbps, result.minimumKbps);

  return TTV_EC_SUCCESS;
}

ttv::broadcast::ArchivingState::ArchivingState() : recordingEnabled(false) {}

ttv::broadcast::GameInfo::GameInfo() : popularity(0), gameId(0) {}
