#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/ingesttester.h"

#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpcontext.h"
#include "twitchsdk/broadcast/internal/streamer.h"
#include "twitchsdk/broadcast/internal/streamstats.h"
#include "twitchsdk/broadcast/ipreencodedvideoframereceiver.h"
#include "twitchsdk/broadcast/ivideocapture.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"
#include "twitchsdk/broadcast/passthroughvideoencoder.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/eventtracker.h"
#include "twitchsdk/core/memory.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/trackingcontext.h"
#include "twitchsdk/core/user/user.h"

#include <condition_variable>

#include <atomic>
#include <mutex>

namespace {
using namespace ttv;
using namespace ttv::broadcast;

const char* kLoggerName = "IngestTester";
const uint32_t kOutputWidth = 416;           // ttv::kMaxFrameWidth;
const uint32_t kOutputHeight = 304;          // ttv::kMaxFrameHeight;
const uint32_t kOutputFramesPerSecond = 60;  //  ttv::kMaxFramesPerSecond;
const uint64_t kTestDurationMilliseconds = 8000;

void UpdateErrorCode(TTV_ErrorCode& var, TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(var)) {
    var = ec;
  }
}
}  // namespace

namespace ttv {
namespace broadcast {
/**
 * A simple video capturer that generates random frames to induce large encoded frames.
 */
class SampleDataVideoCapturer : public IVideoCapture {
 public:
  SampleDataVideoCapturer(const std::shared_ptr<IngestSampleData>& sampleData)
      : mSampleData(sampleData), mDelayState(FrameWriter::DelayState::Okay), mSubmitting(false), mShutdown(false) {}

  virtual ~SampleDataVideoCapturer() override { Shutdown(); }

  // IVideoFrameGenerator implementation
  virtual std::string GetName() const override { return "IngestTesterSampleDataVideoCapturer"; }

  virtual TTV_ErrorCode SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) override {
    if (mSubmitting) {
      return TTV_EC_INVALID_STATE;
    }

    mVideoEncoder = encoder;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode SetFrameQueue(const std::shared_ptr<IVideoFrameQueue>& queue) override {
    if (mSubmitting) {
      return TTV_EC_INVALID_STATE;
    }

    mFrameQueue = queue;

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode Initialize() override { return TTV_EC_SUCCESS; }

  virtual TTV_ErrorCode Start(const VideoParams& videoParams) override {
    ttv::trace::Message("IngestTester", MessageLevel::Debug, "SampleDataVideoCapturer::Start()");

    if (mSubmitting) {
      return TTV_EC_INVALID_STATE;
    }

    if (mVideoEncoder == nullptr || mFrameQueue == nullptr) {
      return TTV_EC_INVALID_STATE;
    }

    // We require a IRawVideoFrameReceiver to submit to
    if (!mVideoEncoder->SupportsReceiverProtocol(IPreEncodedVideoFrameReceiver::GetReceiverTypeId())) {
      return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
    }

    // Extract the IRawVideoFrameReceiver
    auto receiver = mVideoEncoder->GetReceiverImplementation(IPreEncodedVideoFrameReceiver::GetReceiverTypeId());
    TTV_ASSERT(receiver != nullptr);
    mPreEncodedVideoFrameReceiver = std::static_pointer_cast<IPreEncodedVideoFrameReceiver>(receiver);

    mVideoParams = videoParams;

    TTV_ErrorCode ec = CreateThread([this]() { ThreadProc(); }, "ttv::broadcast::SampleDataVideoCapturer", mThread);

    if (TTV_SUCCEEDED(ec)) {
      {
        std::unique_lock<std::mutex> lock(mMutex);
        mSubmitting = true;
        mShutdown = false;
      }
      mCondition.notify_all();

      mThread->Run();
    }

    return ec;
  }

  virtual TTV_ErrorCode Stop() override {
    ttv::trace::Message("IngestTester", MessageLevel::Debug, "SampleDataVideoCapturer::Stop()");

    {
      std::unique_lock<std::mutex> lock(mMutex);
      mShutdown = true;
    }
    mCondition.notify_all();

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode Shutdown() override {
    ttv::trace::Message("IngestTester", MessageLevel::Debug, "SampleDataVideoCapturer::Shutdown()");
    {
      std::unique_lock<std::mutex> lock(mMutex);
      mShutdown = true;
    }
    mCondition.notify_all();

    if (mThread != nullptr) {
      mThread->Join();
      mThread.reset();
    }

    return TTV_EC_SUCCESS;
  }

  void DelayStateChanged(FrameWriter::DelayState state) {
    ttv::trace::Message("IngestTester", MessageLevel::Debug, "SampleDataVideoCapturer::DelayStateChanged(%lu)", state);
    {
      std::unique_lock<std::mutex> lock(mMutex);
      mDelayState = state;
    }
    mCondition.notify_all();
  }

  void Reset() {
    mSubmitting = false;
    mShutdown = false;
    mDelayState = FrameWriter::DelayState::Okay;
  }

 protected:
  VideoParams mVideoParams;

 private:
  void ThreadProc() {
    uint32_t totalFramesSubmitted = 0;
    std::unique_lock<std::mutex> lock(mMutex);
    for (;;) {
      if (mShutdown) {
        break;
      } else if (mDelayState == FrameWriter::DelayState::Okay) {
        // Unlock while actually submitting the frame
        lock.unlock();
        TTV_ErrorCode ec = SubmitFrame(totalFramesSubmitted);

        // TODO: At some point, we should engineer a more direct relationship to avoid this hack.
        // We must make sure we are not submitting frames faster than the video streamer can consume them.
        // Otherwise, the video streamer will end up hanging onto these frames, and we won't know that
        // we've flooded the socket until its too late. The video streamer sleeps 1 ms between processing
        // frames, so we will sleep 2 ms.
        ttv::Sleep(2);
        lock.lock();

        if (TTV_SUCCEEDED(ec)) {
          totalFramesSubmitted++;
        } else {
          mShutdown = true;
          break;
        }
      } else {
        ttv::trace::Message("IngestTester", MessageLevel::Debug,
          "Waiting while queue is delayed. Frames submitted: %lu", totalFramesSubmitted);
        mCondition.wait(lock);
      }
    }

    mSubmitting = false;
    mShutdown = false;
    mDelayState = FrameWriter::DelayState::Okay;

    ttv::trace::Message("IngestTester", MessageLevel::Debug, "VideoCapture thread exiting");
  }

  TTV_ErrorCode SubmitFrame(uint32_t frameIndex) {
    // Package the frame
    std::shared_ptr<VideoFrame> videoFrame;

    const auto& sampleFrame = mSampleData->frames[frameIndex % mSampleData->frames.size()];
    TTV_ErrorCode ec = mPreEncodedVideoFrameReceiver->PackageFrame(
      std::vector<uint8_t>(sampleFrame.data), sampleFrame.keyFrame, static_cast<uint64_t>(frameIndex + 1), videoFrame);

    TTV_ASSERT(TTV_SUCCEEDED(ec));
    TTV_ASSERT(videoFrame != nullptr);

    // Submit for processing
    ttv::trace::Message("IngestTester", MessageLevel::Debug, "Submitting frame");
    ec = mFrameQueue->AddVideoFrame(videoFrame);

    return ec;
  }

  std::shared_ptr<IVideoEncoder> mVideoEncoder;
  std::shared_ptr<IPreEncodedVideoFrameReceiver> mPreEncodedVideoFrameReceiver;
  std::shared_ptr<IVideoFrameQueue> mFrameQueue;
  std::shared_ptr<IThread> mThread;  // The worker thread submitting frames to the frame queue.
  std::shared_ptr<IngestSampleData> mSampleData;
  std::mutex mMutex;
  std::condition_variable mCondition;
  FrameWriter::DelayState mDelayState;
  bool mSubmitting;
  bool mShutdown;
};
}  // namespace broadcast
}  // namespace ttv

ttv::broadcast::IIngestTester::~IIngestTester() {}

ttv::broadcast::IngestTester::IngestTester(
  const std::shared_ptr<User>& user, const std::shared_ptr<StreamerContext>& streamerContext)
    : UserComponent(user),
      mStreamerContext(streamerContext),
      mMeasuredKbps(0),
      mTestErrorCode(TTV_EC_SUCCESS),
      mTestState(TestState::Stopped),
      mTestDurationMilliseconds(kTestDurationMilliseconds),
      mStartingBytesSent(0),
      mTotalBytesSent(0),
      mTotalVideoPacketsSent(0),
      mProgress(0),
      mBroadcasting(false),
      mWaitingForStartCallback(false),
      mWaitingForStopCallback(false) {}

ttv::broadcast::IngestTester::~IngestTester() {
  TTV_ASSERT(IsDone());
}

TTV_ErrorCode ttv::broadcast::IngestTester::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::IngestTester::AddListener(const std::shared_ptr<IIngestTesterListener>& listener) {
  mListeners.AddListener(listener);
}

void ttv::broadcast::IngestTester::RemoveListener(const std::shared_ptr<IIngestTesterListener>& listener) {
  mListeners.RemoveListener(listener);
}

TTV_ErrorCode ttv::broadcast::IngestTester::SetTestData(const uint8_t* buffer, uint32_t length) {
  mSampleData = std::make_shared<IngestSampleData>();
  return mSampleData->Parse(buffer, length);
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetTestState(TestState& result) const {
  result = mTestState;

  return TTV_EC_SUCCESS;
}

std::string ttv::broadcast::IngestTester::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::broadcast::IngestTester::Initialize() {
  Log(MessageLevel::Debug, "IngestTester::Initialize()");

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto user = mUser.lock();
  if (user == nullptr) {
    ec = TTV_EC_NEED_TO_LOGIN;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = UserComponent::Initialize();
  }

  if (TTV_SUCCEEDED(ec)) {
    mVideoCapturer = std::make_shared<SampleDataVideoCapturer>(mSampleData);
    ec = mVideoCapturer->Initialize();
  }

  if (TTV_SUCCEEDED(ec)) {
    // Create a temporary Streamer associated with the user but don't hang it off of the user
    mStreamer = std::make_shared<Streamer>(user, mStreamerContext);
    mStreamer->SetVideoCapturer(mVideoCapturer);

    mStreamerListener = std::make_shared<StreamerListenerProxy>();

    mStreamerListener->mStreamerStateChangedFunc = [this](Streamer* /*source*/, Streamer::StreamerState state,
                                                     TTV_ErrorCode changedEc) {
      switch (state) {
        case Streamer::StreamerState::Stopped:
          // Unsolicited stop
          if (!IsDone() && TTV_FAILED(changedEc)) {
            mTestErrorCode = changedEc;
            OnStreamerStopped();
          }
          break;
        default:
          break;
      }
    };

    mStreamer->AddListener(mStreamerListener);

    mStreamer->SetFrameWriterDelayStateChangedCallback(
      [this](FrameWriter::DelayState state) { mVideoCapturer->DelayStateChanged(state); });

    ec = mStreamer->Initialize();
  }

  if (TTV_SUCCEEDED(ec)) {
    mStatsListener = std::make_shared<StreamStatsListenerProxy>();

    auto stats = mStreamer->GetStreamStats();
    TTV_ASSERT(stats != nullptr);
    stats->AddListener(mStatsListener);

    mStatsListener->mOnStatReceivedFunc = [this](StreamStats* /*source*/, StreamStats::StatType type, uint64_t data) {
      switch (type) {
        case StreamStats::StatType::RtmpState: {
          // Start the timer to measure bandwidth
          RtmpContext::State state = static_cast<RtmpContext::State>(data);
          if (state == RtmpContext::State::Publish) {
            mServerTestTimer.Set(mTestDurationMilliseconds);
          }
          break;
        }
        case StreamStats::StatType::RtmpTotalBytesSent: {
          if (mServerTestTimer.IsSet()) {
            mTotalBytesSent = data - mStartingBytesSent;
          } else {
            mStartingBytesSent = data;
          }
          break;
        }
        case StreamStats::StatType::TotalVideoPacketsSent: {
          mTotalVideoPacketsSent = data;
          ttv::trace::Message(
            "IngestTester", MessageLevel::Debug, "TotalVideoPacketsSent: %d", static_cast<uint32_t>(data));

          break;
        }
        default: { break; }
      }
    };
  }

  if (TTV_SUCCEEDED(ec)) {
    mComponentContainer = std::make_shared<ComponentContainer>();
    ec = mComponentContainer->Initialize();
    TTV_ASSERT(TTV_SUCCEEDED(ec));

    ec = mComponentContainer->AddComponent(mStreamer);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::IngestTester::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mVideoCapturer != nullptr) {
      mVideoCapturer->Shutdown();
    }
  }

  return ec;
}

bool ttv::broadcast::IngestTester::CheckShutdown() {
  if (!UserComponent::CheckShutdown()) {
    return false;
  }

  if (mComponentContainer != nullptr) {
    if (mComponentContainer->GetState() == ComponentContainer::State::Initialized) {
      mComponentContainer->Shutdown();
    }

    if (mComponentContainer->GetState() != ComponentContainer::State::Uninitialized) {
      return false;
    }
  }

  return IsDone();
}

void ttv::broadcast::IngestTester::CompleteShutdown() {
  if (mStreamer != nullptr) {
    auto stats = mStreamer->GetStreamStats();
    if (stats != nullptr) {
      stats->RemoveListener(mStatsListener);
    }

    mStreamer.reset();
  }

  mMeasuredKbps = 0;
  mProgress = 0;
  mTestErrorCode = TTV_EC_SUCCESS;
  mListeners.ClearListeners();
  mStatsListener.reset();
  mStreamerListener.reset();
  mVideoCapturer.reset();
  mVideoEncoder.reset();
  mComponentContainer.reset();

  UserComponent::CompleteShutdown();
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetIngestServer(IngestServer& result) const {
  result = mIngestServer;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::Start(const IngestServer& server) {
  if (!IsDone()) {
    return TTV_EC_INVALID_STATE;
  } else if (server.serverUrl.empty()) {
    return TTV_EC_BROADCAST_INVALID_INGEST_SERVER;
  }

  Log(MessageLevel::Debug, "IngestTester::Start()");

  mVideoCapturer->Reset();

  auto videoEncoder = std::make_shared<PassThroughVideoEncoder>();
  videoEncoder->SetSps(mSampleData->spsData);
  videoEncoder->SetPps(mSampleData->ppsData);
  mVideoEncoder = std::move(videoEncoder);

  TTV_ErrorCode ec = mStreamer->SetVideoEncoder(mVideoEncoder);

  if (TTV_SUCCEEDED(ec)) {
    VideoParams::ConfigureForResolution(
      kOutputWidth, kOutputHeight, kOutputFramesPerSecond, kRecommendedBitsPerPixel, mVideoParams);
    mVideoParams.automaticBitRateAdjustmentEnabled = false;
  }

  if (TTV_SUCCEEDED(ec)) {
    mIngestServer = server;
    mMeasuredKbps = 0;
    mTestErrorCode = ec;

    ec = StartServerTest();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::IngestTester::Cancel() {
  if (mTestState == TestState::Stopped) {
    return TTV_EC_INVALID_STATE;
  }

  Log(MessageLevel::Debug, "IngestTester::Cancel()");

  if (!IsDone()) {
    UpdateErrorCode(mTestErrorCode, TTV_EC_REQUEST_ABORTED);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetMeasuredKbps(uint32_t& result) {
  result = mMeasuredKbps;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetTestError(TTV_ErrorCode& result) {
  result = mTestErrorCode;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::SetTestDurationMilliseconds(uint64_t duration) {
  if (duration == 0) {
    return TTV_EC_INVALID_ARG;
  } else if (IsDone()) {
    mTestDurationMilliseconds = duration;

    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_INVALID_STATE;
  }
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetTestDurationMilliseconds(uint64_t& result) const {
  result = mTestDurationMilliseconds;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetProgress(float& result) const {
  result = mProgress;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IngestTester::GetUserId(UserId& result) const {
  auto user = mUser.lock();

  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  result = user->GetUserId();

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::IngestTester::SetTestState(TestState state) {
  if (state == mTestState) {
    return;
  }

  mTestState = state;

  mListeners.Invoke(
    [this](std::shared_ptr<IIngestTesterListener> listener) { listener->BroadcastIngestTesterStateChanged(this); });
}

bool ttv::broadcast::IngestTester::IsDone() const {
  switch (mTestState) {
    case TestState::Stopped:
    case TestState::Finished:
    case TestState::Failed:
      return true;
    default:
      return false;
  }
}

void ttv::broadcast::IngestTester::Update() {
  UserComponent::Update();

  if (mComponentContainer != nullptr) {
    mComponentContainer->Update();
  }

  if (mState != State::Initialized) {
    return;
  }

  // Wait for callbacks to come in
  if (mWaitingForStartCallback || mWaitingForStopCallback) {
    return;
  }

  switch (mTestState) {
    case TestState::Testing: {
      UpdateServerTest();
      break;
    }
    case TestState::Disconnecting: {
      if (TTV_FAILED(mTestErrorCode)) {
        mMeasuredKbps = 0;
      }

      StopServerTest();

      break;
    }
    // Idle states
    case TestState::Connecting:
    case TestState::Stopped:
    case TestState::Finished:
    case TestState::Failed: {
      break;
    }
    default: { break; }
  }
}

TTV_ErrorCode ttv::broadcast::IngestTester::StartServerTest() {
  // Reset the test
  mTestErrorCode = TTV_EC_SUCCESS;
  mStartingBytesSent = 0;
  mTotalBytesSent = 0;
  mTotalVideoPacketsSent = 0;
  mMeasuredKbps = 0;
  mServerTestTimer.Clear();

  // Start the stream
  mWaitingForStartCallback = true;

  Streamer::StartParams startParams;
  startParams.videoParams = mVideoParams;
  startParams.ingestServer = mIngestServer;
  startParams.flags = StreamStartFlags::BandwidthTest;
  startParams.enableAsyncFlvMuxer = false;

  mStreamerContext->sharedTrackingContext->TrackEvent("mobile_broadcast_ingest_test_initiated",
    {{"ingest_server_id", mIngestServer.serverId}, {"ingest_server_name", mIngestServer.serverName}});

  TTV_ErrorCode ec = mStreamer->Start(startParams, [this](TTV_ErrorCode callbackEc) {
    mWaitingForStartCallback = false;

    UpdateErrorCode(mTestErrorCode, callbackEc);

    if (TTV_SUCCEEDED(callbackEc)) {
      OnStreamerStarted();
    } else {
      OnStreamerStopped();
    }
  });

  if (TTV_SUCCEEDED(ec)) {
    SetTestState(TestState::Connecting);
  } else {
    mStreamerContext->sharedTrackingContext->TrackEvent(
      "mobile_broadcast_ingest_test_failed", {{"error_code", ttv::ErrorToString(ec)}, {"synchronous", true}});
    mWaitingForStartCallback = false;
  }

  return ec;
}

void ttv::broadcast::IngestTester::UpdateServerTest() {
  // Check for error or timeout
  if (TTV_FAILED(mTestErrorCode) || mServerTestTimer.Check(false)) {
    SetTestState(TestState::Disconnecting);
  }
  // Connected and sending video
  else if (mBroadcasting && mServerTestTimer.IsSet()) {
    uint64_t millisecondsElapsed = mServerTestTimer.GetElapsedTime();
    if (millisecondsElapsed > 0) {
      uint64_t totalBits = 8 * mTotalBytesSent;
      mMeasuredKbps = static_cast<uint32_t>(
        totalBits / millisecondsElapsed);  // Conversion to kilobits and to seconds cancel each other out
    }
  }
}

void ttv::broadcast::IngestTester::StopServerTest() {
  if (mWaitingForStartCallback) {
    // Wait for the start callback and do the stop after that comes in
    UpdateErrorCode(mTestErrorCode, TTV_EC_REQUEST_ABORTED);
  } else if (mBroadcasting) {
    if (!mWaitingForStopCallback) {
      mWaitingForStopCallback = true;

      mVideoCapturer->Stop();
      TTV_ErrorCode ec = mStreamer->Stop("ingest_test", [this](TTV_ErrorCode callbackEc) {
        mWaitingForStopCallback = false;

        TTV_ASSERT(TTV_SUCCEEDED(callbackEc));
        OnStreamerStopped();
      });

      if (TTV_FAILED(ec)) {
        // This should never happen so fake the callback to indicate it's stopped
        OnStreamerStopped();

        Log(MessageLevel::Error, "IngestTester::StopServerTest() - Stop failed");
      }
    }
  } else {
    OnStreamerStopped();
  }
}

void ttv::broadcast::IngestTester::OnStreamerStarted() {
  mBroadcasting = true;
  mWaitingForStartCallback = false;

  // While starting there was an error so stop the stream
  if (TTV_FAILED(mTestErrorCode)) {
    mWaitingForStopCallback = true;

    TTV_ErrorCode ec = mStreamer->Stop("ingest_test", [this](TTV_ErrorCode callbackEc) {
      mWaitingForStopCallback = false;

      TTV_ASSERT(TTV_SUCCEEDED(callbackEc));
      OnStreamerStopped();
    });

    if (TTV_FAILED(ec)) {
      mWaitingForStopCallback = false;

      UpdateErrorCode(mTestErrorCode, ec);

      // This should never happen so fake the callback to indicate it's stopped
      OnStreamerStopped();

      Log(MessageLevel::Error, "IngestTester::StopServerTest() - Stop failed");
    }
  } else {
    SetTestState(TestState::Testing);
  }
}

void ttv::broadcast::IngestTester::OnStreamerStopped() {
  mBroadcasting = false;

  if (TTV_SUCCEEDED(mTestErrorCode)) {
    mStreamerContext->sharedTrackingContext->TrackEvent("mobile_broadcast_ingest_test_completed",
      {{"ingest_server_id", mIngestServer.serverId}, {"ingest_server_name", mIngestServer.serverName},
        {"ingest_kbps", mMeasuredKbps}});
    SetTestState(TestState::Finished);
  } else {
    mStreamerContext->sharedTrackingContext->TrackEvent(
      "mobile_broadcast_ingest_test_completed", {{"error_code", mTestErrorCode}, {"synchronous", false}});
    SetTestState(TestState::Failed);
  }
}

void ttv::broadcast::IngestTester::UpdateProgress() {
  switch (mTestState) {
    case TestState::Finished:
    case TestState::Failed: {
      mProgress = 1.0f;
      break;
    }
    case TestState::Testing: {
      mProgress = static_cast<float>(mServerTestTimer.GetElapsedTime()) / static_cast<float>(mTestDurationMilliseconds);
      break;
    }
    case TestState::Stopped:
    case TestState::Connecting:
    default: {
      mProgress = 0.0f;
      break;
    }
  }
}
