/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/muxers/amf0.h"

namespace {
using namespace ttv::broadcast;

const uint8_t* DecodeElement(const uint8_t* data, std::shared_ptr<IAMF0> output);

template <typename result_t>
result_t ConvertTo(const uint8_t*& data) {
  result_t result = 0;
  auto p = reinterpret_cast<uint8_t*>(&result);

  for (size_t i = 0; i < sizeof(result_t); i++) {
    p[i] = data[sizeof(result_t) - i - 1];
  }

  data += sizeof(result_t);
  return result;
}

template <>
std::string ConvertTo(const uint8_t*& data) {
  auto length = ConvertTo<uint16_t>(data);

  std::string param(data, data + length);

  data += length;

  return param;
}

const uint8_t* DecodeString(const uint8_t* data, std::shared_ptr<IAMF0> output, IAMF0::AMFType kind) {
  assert(kind == IAMF0::string || kind == IAMF0::object || kind == IAMF0::ecmaArray);

  auto param = ConvertTo<std::string>(data);

  switch (kind) {
    case IAMF0::string:
      output->String(param);
      break;
    case IAMF0::object:
      output->ObjectProperty(param);
      break;
    case IAMF0::ecmaArray:
      output->EcmaArrayKey(param);
      break;
    default:
      // should never get here without asserting
      break;
  }
  return data;
}

const uint8_t* DecodeDate(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto value = ConvertTo<double>(data);
  auto timeZome = ConvertTo<int16_t>(data);
  TTV_ASSERT(timeZome == 0);

  output->Date(value);
  return data;
}

const uint8_t* DecodeNumber(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto value = ConvertTo<double>(data);

  output->Number(value);
  return data;
}

const uint8_t* DecodeEmcaArray(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto elements = ConvertTo<uint32_t>(data);
  output->EcmaArray(elements);

  for (uint32_t i = 0; i < elements; i++) {
    data = DecodeString(data, output, IAMF0::ecmaArray);
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeStrictArray(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto elements = ConvertTo<uint32_t>(data);
  output->StrictArray(elements);

  for (uint32_t i = 0; i < elements; i++) {
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeObject(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  output->Object();

  for (;;) {
    if (data[0] == 0 && data[1] == 0 && data[2] == IAMF0::objectEnd) {
      output->ObjectEnd();
      data += 3;
      break;
    }
    data = DecodeString(data, output, IAMF0::object);
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeBoolean(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  bool flag = (*data++ != 0);
  output->Boolean(flag);
  return data;
}

const uint8_t* DecodeElement(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  IAMF0::AMFType type = static_cast<IAMF0::AMFType>(*data++);

  assert(type != IAMF0::objectEnd);
  assert(type != IAMF0::movieclip);
  assert(type != IAMF0::recordset);
  assert(type != IAMF0::unsupported);

  switch (type) {
    case IAMF0::number:
      data = DecodeNumber(data, output);
      break;
    case IAMF0::boolean:
      data = DecodeBoolean(data, output);
      break;
    case IAMF0::string:
      data = DecodeString(data, output, ttv::broadcast::IAMF0::string);
      break;
    case IAMF0::object:
      data = DecodeObject(data, output);
      break;
    case IAMF0::null:
      output->Null();
      break;
    case IAMF0::undefined:
      output->Undefined();
      assert(false && "untested");
      break;
    case IAMF0::reference:
      output->Reference();
      assert(false && "unimplemented");
      break;
    case IAMF0::ecmaArray:
      data = DecodeEmcaArray(data, output);
      break;
    case IAMF0::strictArray:
      data = DecodeStrictArray(data, output);
      break;
    case IAMF0::date:
      data = DecodeDate(data, output);
      break;
    case IAMF0::longString:
      output->LongString();
      assert(false && "unimplemented");
      break;
    case IAMF0::xmlDocument:
      output->XmlDocument();
      break;
    case IAMF0::typedObject:
      output->TypedObject();
      assert(false && "unimplemented");
      break;
    default:
      assert(false);
      break;
  }

  return data;
}
}  // namespace

ttv::broadcast::IAMF0::~IAMF0() {}

void ttv::broadcast::AMF0Nop::Number(double /*value*/) {}
void ttv::broadcast::AMF0Nop::Boolean(bool /*flag*/) {}
void ttv::broadcast::AMF0Nop::String(std::string /*param*/) {}
void ttv::broadcast::AMF0Nop::Object() {}
void ttv::broadcast::AMF0Nop::ObjectProperty(std::string /*propertyName*/) {}
void ttv::broadcast::AMF0Nop::Movieclip() {}
void ttv::broadcast::AMF0Nop::Null() {}
void ttv::broadcast::AMF0Nop::Undefined() {}
void ttv::broadcast::AMF0Nop::Reference() {}
void ttv::broadcast::AMF0Nop::EcmaArray(uint32_t /*elements*/) {}
void ttv::broadcast::AMF0Nop::EcmaArrayKey(std::string /*keyName*/) {}
void ttv::broadcast::AMF0Nop::ObjectEnd() {}
void ttv::broadcast::AMF0Nop::StrictArray(uint32_t /*elements*/) {}
void ttv::broadcast::AMF0Nop::Date(double /*dateValue*/) {}
void ttv::broadcast::AMF0Nop::LongString() {}
void ttv::broadcast::AMF0Nop::Unsupported() {}
void ttv::broadcast::AMF0Nop::Recordset() {}
void ttv::broadcast::AMF0Nop::XmlDocument() {}
void ttv::broadcast::AMF0Nop::TypedObject() {}

const uint8_t* ttv::broadcast::DecodeAMF(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  assert(data);
  assert(output);

  return DecodeElement(data, output);
}

ttv::broadcast::AMF0StringDecoder::AMF0StringDecoder() {}

void ttv::broadcast::AMF0StringDecoder::String(std::string param) {
  if (mCommandName.length() == 0) {
    mCommandName = param;
  }
}

ttv::broadcast::AMF0PropertyDecoder::AMF0PropertyDecoder(const std::string& searchTerm)
    : mSearchTerm(searchTerm), mFoundProperty(0) {}

void ttv::broadcast::AMF0PropertyDecoder::String(std::string param) {
  if (mFoundProperty) {
    mFoundValue = param;
    mFoundProperty = false;
  }
}

void ttv::broadcast::AMF0PropertyDecoder::ObjectProperty(std::string propertyName) {
  mFoundProperty = strcasecmp(propertyName.c_str(), mSearchTerm.c_str()) == 0;
}

ttv::broadcast::AMF0NumberDecoder::AMF0NumberDecoder() : mValue(0.0) {}

void ttv::broadcast::AMF0NumberDecoder::Number(double value) {
  mValue = value;
}
