/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/muxers/flvmuxerasync.h"

#include "twitchsdk/broadcast/internal/muxers/amf0encoder.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpstream.h"
#include "twitchsdk/broadcast/internal/streamstats.h"
#include "twitchsdk/broadcast/packet.h"

#include <codecvt>
#include <functional>
#include <iostream>

ttv::broadcast::FlvMuxerAsync::FlvMuxerAsync(std::shared_ptr<StreamStats> streamStats)
    : FlvMuxer(streamStats), mDoProcessing(false) {}

ttv::broadcast::FlvMuxerAsync::~FlvMuxerAsync() {
  Stop();
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::Start(const MuxerParameters& params) {
  StartProcessThread();

  ExecuteAsync([this, params]() { return FlvMuxer::Start(params); });

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::FlvMuxerAsync::Update() {
  ExecuteAsync([this] {
    FlvMuxer::Update();
    return TTV_EC_SUCCESS;
  });
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::Stop() {
  if (mDoProcessing) {
    ExecuteSyncWithResult([this] { return FlvMuxer::Stop(); });
  }

  StopProcessThread();
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::WriteVideoPacket(const Packet& packet) {
  TTV_ASSERT(packet.timestamp <= UINT_MAX);

  ExecuteAsync([this, packet] { return FlvMuxer::WriteVideoPacket(packet); });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::WriteAudioPacket(const Packet& packet) {
  TTV_ASSERT(packet.timestamp <= UINT_MAX);

  ExecuteAsync([this, packet] { return FlvMuxer::WriteAudioPacket(packet); });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::GetError() {
  TTV_ErrorCode ec = ExecuteSyncWithResult([this]() { return FlvMuxer::GetError(); });
  return ec;
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::GetAverageSendBitRate(
  uint64_t measurementWindowMilliseconds, uint64_t& bitsPerSecond) const {
  TTV_ErrorCode ec = ExecuteSyncWithResult([this, measurementWindowMilliseconds, &bitsPerSecond]() -> TTV_ErrorCode {
    return FlvMuxer::GetAverageSendBitRate(measurementWindowMilliseconds, bitsPerSecond);
  });
  return ec;
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::GetCongestionLevel(
  uint64_t measurementWindowMilliseconds, double& congestionLevel) const {
  TTV_ErrorCode ec = ExecuteSyncWithResult([this, measurementWindowMilliseconds, &congestionLevel]() -> TTV_ErrorCode {
    return FlvMuxer::GetCongestionLevel(measurementWindowMilliseconds, congestionLevel);
  });
  return ec;
}

void ttv::broadcast::FlvMuxerAsync::ExecuteAsync(ExecuteFunction executeFunction) const {
  std::shared_ptr<Task> task = std::make_shared<Task>(executeFunction);
  ExecuteTask(task);
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::ExecuteSyncWithResult(ExecuteFunction executeFunction) const {
  std::shared_ptr<Task> task = std::make_shared<Task>(executeFunction);
  ExecuteTask(task);
  return task->GetResultFromFuture();
}

void ttv::broadcast::FlvMuxerAsync::ExecuteTask(std::shared_ptr<Task> task) const {
  {
    std::lock_guard<std::mutex> lock(mProcessQueueMutex);
    mProcessQueue.push(task);
  }
  mProcessQueueCond.notify_one();
}

void ttv::broadcast::FlvMuxerAsync::StartProcessThread() {
  if (!mProcessThread) {
    TTV_ErrorCode ec = ttv::CreateThread(
      std::bind(&FlvMuxerAsync::ProcessQueue, this), "ttv::broadcast::FlvMuxerAsync::RtmpThread", mProcessThread);
    TTV_ASSERT(TTV_SUCCEEDED(ec) && mProcessThread != nullptr);
    mDoProcessing = true;
    mProcessThread->Run();
  }
}

void ttv::broadcast::FlvMuxerAsync::StopProcessThread() {
  if (!mDoProcessing) {
    return;
  }

  mDoProcessing = false;
  mProcessQueueCond.notify_all();
  mProcessThread->Join();
}

void ttv::broadcast::FlvMuxerAsync::ProcessQueue() {
  while (mDoProcessing) {
    std::shared_ptr<Task> task;

    {
      std::unique_lock<std::mutex> lock(mProcessQueueMutex);
      mProcessQueueCond.wait(lock, [this]() { return !mProcessQueue.empty() || !mDoProcessing; });

      if (!mProcessQueue.empty() && mDoProcessing) {
        task = mProcessQueue.front();
        mProcessQueue.pop();
      }
    }

    if (task) {
      task->Run();
    }
  }
}

ttv::broadcast::FlvMuxerAsync::Task::Task(ExecuteFunction func) : mExecuteFunction(func) {}

void ttv::broadcast::FlvMuxerAsync::Task::Run() {
  TTV_ErrorCode result = mExecuteFunction();
  mResultPromise.set_value(result);
}

TTV_ErrorCode ttv::broadcast::FlvMuxerAsync::Task::GetResultFromFuture() {
  return mResultPromise.get_future().get();
}
