/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/muxers/rtmpcreatestreamstate.h"

TTV_ErrorCode ttv::broadcast::RtmpCreateStreamState::SendOutgoingChunkSizeMessage() {
  uint32_t newSize = BigToLittle(RtmpContext::kDesiredOutgoingChunkSize);

  RtmpMessageDetails details(RTMP_CHANNEL_NETWORK, 0, sizeof(newSize), RTMP_PKT_CHUNK_SIZE, 0);

  auto buffer = reinterpret_cast<uint8_t*>(&newSize);
  TTV_ErrorCode ret = SendChunkData(buffer, sizeof(newSize), details);
  ASSERT_ON_ERROR(ret);

  GetContext()->mOutgoingChunkSize = RtmpContext::kDesiredOutgoingChunkSize;

  return ret;
}

TTV_ErrorCode ttv::broadcast::RtmpCreateStreamState::SendReleaseStreamMessage() {
  GetContext()->mAmfEncoder.Reset();

  GetContext()->mAmfEncoder.String("releaseStream");
  GetContext()->mAmfEncoder.Number(GetContext()->mTransactionId++);
  GetContext()->mAmfEncoder.Null();
  GetContext()->mAmfEncoder.String(GetContext()->mStreamName);

  RtmpMessageDetails details(
    RTMP_CHANNEL_SYSTEM, 0, static_cast<uint32_t>(GetContext()->mAmfEncoder.GetBuffer().size()), RTMP_PKT_AMF0, 0);

  TTV_ErrorCode ret = SendChunkData(GetContext()->mAmfEncoder.GetBuffer(), details);
  ASSERT_ON_ERROR(ret);

  return ret;
}

TTV_ErrorCode ttv::broadcast::RtmpCreateStreamState::SendFCPublishMessage() {
  GetContext()->mAmfEncoder.Reset();

  GetContext()->mAmfEncoder.String("FCPublish");
  GetContext()->mAmfEncoder.Number(GetContext()->mTransactionId++);
  GetContext()->mAmfEncoder.Null();
  GetContext()->mAmfEncoder.String(GetContext()->mStreamName);

  RtmpMessageDetails details(
    RTMP_CHANNEL_SYSTEM, 0, static_cast<uint32_t>(GetContext()->mAmfEncoder.GetBuffer().size()), RTMP_PKT_AMF0, 0);

  TTV_ErrorCode ret = SendChunkData(GetContext()->mAmfEncoder.GetBuffer(), details);
  ASSERT_ON_ERROR(ret);

  return ret;
}

TTV_ErrorCode ttv::broadcast::RtmpCreateStreamState::SendCreateStreamMessage() {
  GetContext()->mAmfEncoder.Reset();

  GetContext()->mAmfEncoder.String("createStream");
  GetContext()->mAmfEncoder.Number(GetContext()->mTransactionId++);
  GetContext()->mAmfEncoder.Null();

  RtmpMessageDetails details(
    RTMP_CHANNEL_SYSTEM, 0, static_cast<uint32_t>(GetContext()->mAmfEncoder.GetBuffer().size()), RTMP_PKT_AMF0, 0);

  TTV_ErrorCode ret = SendChunkData(GetContext()->mAmfEncoder.GetBuffer(), details);
  ASSERT_ON_ERROR(ret);

  return ret;
}

void ttv::broadcast::RtmpCreateStreamState::OnEnterInternal() {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  ret = SendOutgoingChunkSizeMessage();
  if (TTV_FAILED(ret)) {
    GetContext()->SetNextState(RtmpContext::State::Error);
    GetContext()->mLastError = ret;
    return;
  }

  ret = SendReleaseStreamMessage();
  if (TTV_FAILED(ret)) {
    GetContext()->SetNextState(RtmpContext::State::Error);
    GetContext()->mLastError = ret;
    return;
  }

  ret = SendFCPublishMessage();
  if (TTV_FAILED(ret)) {
    GetContext()->SetNextState(RtmpContext::State::Error);
    GetContext()->mLastError = ret;
    return;
  }

  ret = SendCreateStreamMessage();
  if (TTV_FAILED(ret)) {
    GetContext()->SetNextState(RtmpContext::State::Error);
    GetContext()->mLastError = ret;
    return;
  }

  ret = GetContext()->mSocket.FlushCache();
  if (TTV_FAILED(ret)) {
    GetContext()->SetNextState(RtmpContext::State::Error);
    GetContext()->mLastError = ret;
    return;
  }
}

void ttv::broadcast::RtmpCreateStreamState::HandleIncomingAmf0(ChunkHeader /*header*/, const uint8_t* data) {
  std::shared_ptr<AMF0StringDecoder> stringDecoder(new AMF0StringDecoder());
  data = DecodeAMF(data, stringDecoder);

  // Skip over the number
  std::shared_ptr<AMF0Nop> nopDecoder(new AMF0Nop());
  data = DecodeAMF(data, nopDecoder);

  // command meta data
  data = DecodeAMF(data, nopDecoder);

  if (strcasecmp(stringDecoder->GetCommandName().c_str(), "_result") == 0) {
    std::shared_ptr<AMF0NumberDecoder> numberDecoder(new AMF0NumberDecoder());
    data = DecodeAMF(data, numberDecoder);

    if (numberDecoder->GetValue() > 0) {
      GetContext()->mStreamId = numberDecoder->GetValue();
      GetContext()->SetNextState(RtmpContext::State::Publish);
    } else {
      ttv::trace::Message("rtmp", MessageLevel::Error, "Unexpected result returned during stream create!");
      GetContext()->SetNextState(RtmpContext::State::Error);
    }
  }

  (void)data;  // We don't care about data's value
}
